"use client"

import { useState, useEffect } from "react"
import { Link } from "react-router-dom"
import { appService } from "../services/api"
import { useAuth } from "../contexts/AuthContext"
import "../styles/home.css"

export const Home = () => {
  const { isAuthenticated, logout, isLoading } = useAuth()
  const [serverStatus, setServerStatus] = useState<"loading" | "online" | "offline">("loading")
  const [featureCards] = useState([
    {
      title: "Chart Analysis",
      description: "Upload and analyze your charts with our powerful tools",
      path: "/charts/analyze",
      public: true,
    },
    {
      title: "Ask Questions",
      description: "Get insights about your data by asking natural language questions",
      path: "/charts/ask",
      public: true,
    },
    {
      title: "Interactive Chat",
      description: "Have a conversation with AI about your charts (Login required)",
      path: "/charts/chat",
      public: false,
    },
  ])

  useEffect(() => {
    const checkServerStatus = async () => {
      try {
        await appService.getRoot()
        setServerStatus("online")
      } catch (error) {
        setServerStatus("offline")
      }
    }

    checkServerStatus()
  }, [])

  const handleLogout = () => {
    logout()
  }

  if (isLoading) {
    return (
      <div className="home-container">
        <div className="loading-spinner">Loading...</div>
      </div>
    )
  }

  return (
    <div className="home-container">
      <header className="home-header">
        <div className="server-status">
          Server Status:
          <span className={`status-indicator ${serverStatus}`}>{serverStatus.toUpperCase()}</span>
        </div>

        <nav className="auth-nav">
          {isAuthenticated ? (
            <div className="user-section">
              <span className="welcome-text">Welcome back!</span>
              <button onClick={handleLogout} className="logout-button">
                Logout
              </button>
            </div>
          ) : (
            <div className="auth-links">
              <Link to="/auth/login" className="auth-link login">
                Login
              </Link>
              <Link to="/auth/register" className="auth-link register">
                Register
              </Link>
            </div>
          )}
        </nav>
      </header>

      <main className="home-main">
        <section className="hero-section">
          <h1>Data Visualization & Analysis Platform</h1>
          <p>
            Transform your data into insights with our powerful chart analysis tools and AI-powered question answering
            system.
          </p>
        </section>

        <section className="features-section">
          <h2>Key Features</h2>
          <div className="feature-cards">
            {featureCards.map((feature, index) => (
              <div key={index} className="feature-card">
                <h3>{feature.title}</h3>
                <p>{feature.description}</p>
                {feature.public || isAuthenticated ? (
                  <Link to={feature.path} className="feature-link">
                    Try it out →
                  </Link>
                ) : (
                  <Link to="/auth/login" className="feature-link login-required">
                    Login to access →
                  </Link>
                )}
              </div>
            ))}
          </div>
        </section>

        {!isAuthenticated && (
          <section className="cta-section">
            <h2>Ready to get started?</h2>
            <p>Create an account to access all features including interactive chat!</p>
            <div className="cta-buttons">
              <Link to="/auth/register" className="cta-button primary">
                Create Account
              </Link>
              <Link to="/auth/login" className="cta-button secondary">
                Login
              </Link>
            </div>
          </section>
        )}
      </main>

      <footer className="home-footer">
        <p>© {new Date().getFullYear()} Data Analysis Platform. All rights reserved.</p>
      </footer>
    </div>
  )
}
