from src.domain.entities.chart_analysis import ChartAnalysis
from src.domain.ports.repositories.analysis_repository import AnalysisRepositoryPort
from src.application.ports.llm_service_port import LLMServicePort
from src.application.dtos.LLM import LLMRequestDTO
import uuid
from datetime import datetime, timezone
from typing import List

class ChatConversationUseCase:
    def __init__(
        self, 
        analysis_repo: AnalysisRepositoryPort,
        llm_service: LLMServicePort
    ):
        self._analysis_repo = analysis_repo
        self._llm_service = llm_service

    def execute(self, chart_image_id: str, conversation_history: List[str], new_question: str) -> ChartAnalysis:
        """Handle interactive chat conversation about a chart"""
        
        # Build context from conversation history
        context = self._build_conversation_context(conversation_history)
        
        # Create enhanced question with context
        enhanced_question = f"Context from previous conversation: {context}\n\nNew question: {new_question}"
        
        # Create request for LLM
        request = LLMRequestDTO(
            image_bytes=b"",  # Will be loaded by service
            question=enhanced_question
        )
        
        # Get response from LLM
        response = await self._llm_service.analyze(request)
        
        # Save the conversation turn
        analysis = ChartAnalysis(
            id=str(uuid.uuid4()),
            chart_image_id=chart_image_id,
            question=new_question,
            answer=response.answer,
            created_at=datetime.now(timezone.utc)
        )
        
        self._analysis_repo.save_analysis(analysis)
        return analysis
    
    def _build_conversation_context(self, history: List[str]) -> str:
        """Build context string from conversation history"""
        if not history:
            return ""
        
        context_parts = []
        for i, entry in enumerate(history[-5:], 1):  # Last 5 entries for context
            context_parts.append(f"Turn {i}: {entry}")
        
        return " | ".join(context_parts) 