from fastapi import APIRouter, Depends, HTTPException, status
from src.application.dtos.conversation import (
    CreateConversationRequestDTO,
    SendMessageRequestDTO,
    ConversationResponseDTO,
    ConversationDetailResponseDTO
)
from src.application.use_cases.chat_conversation import ChatConversationUseCase
from src.dependencies import get_current_user
from typing import List
import logging

logger = logging.getLogger(__name__)
router = APIRouter(tags=["conversations"])

@router.post("/", response_model=ConversationResponseDTO, status_code=status.HTTP_201_CREATED)
async def create_conversation(
    request: CreateConversationRequestDTO,
    current_user: dict = Depends(get_current_user)
):
    """Create a new conversation for a chart"""
    try:
        # Implementation would use conversation service
        # For now, return mock response
        return ConversationResponseDTO(
            id="conv_123",
            title=request.title,
            created_at="2024-01-01T00:00:00Z",
            updated_at="2024-01-01T00:00:00Z",
            message_count=0
        )
    except Exception as e:
        logger.error(f"Error creating conversation: {str(e)}")
        raise HTTPException(status_code=500, detail="Failed to create conversation")

@router.get("/", response_model=List[ConversationResponseDTO])
async def list_conversations(
    current_user: dict = Depends(get_current_user),
    limit: int = 20
):
    """List user's conversations"""
    try:
        # Implementation would fetch from repository
        return []
    except Exception as e:
        logger.error(f"Error listing conversations: {str(e)}")
        raise HTTPException(status_code=500, detail="Failed to list conversations")

@router.get("/{conversation_id}", response_model=ConversationDetailResponseDTO)
async def get_conversation(
    conversation_id: str,
    current_user: dict = Depends(get_current_user)
):
    """Get conversation details with messages"""
    try:
        # Implementation would fetch from repository
        return ConversationDetailResponseDTO(
            id=conversation_id,
            title="Sample Conversation",
            created_at="2024-01-01T00:00:00Z",
            updated_at="2024-01-01T00:00:00Z",
            messages=[]
        )
    except Exception as e:
        logger.error(f"Error getting conversation: {str(e)}")
        raise HTTPException(status_code=500, detail="Failed to get conversation")

@router.post("/{conversation_id}/messages", response_model=MessageResponseDTO)
async def send_message(
    conversation_id: str,
    request: SendMessageRequestDTO,
    current_user: dict = Depends(get_current_user)
):
    """Send a message in a conversation"""
    try:
        # Implementation would use ChatConversationUseCase
        return MessageResponseDTO(
            id="msg_123",
            message_type="assistant",
            content="This is a sample response from the AI.",
            timestamp="2024-01-01T00:00:00Z"
        )
    except Exception as e:
        logger.error(f"Error sending message: {str(e)}")
        raise HTTPException(status_code=500, detail="Failed to send message") 