from abc import ABC, abstractmethod
from src.domain.entities.conversation import Conversation, ConversationMessage
from typing import List, Optional

class ConversationRepositoryPort(ABC):
    """Port interface for conversation repository operations"""
    
    @abstractmethod
    async def create_conversation(self, conversation: Conversation) -> Conversation:
        """Create a new conversation"""
        pass
    
    @abstractmethod
    async def get_conversation_by_id(self, conversation_id: str) -> Optional[Conversation]:
        """Get conversation by ID"""
        pass
    
    @abstractmethod
    async def get_conversations_by_user_id(self, user_id: str, limit: int = 20) -> List[Conversation]:
        """Get all conversations for a user"""
        pass
    
    @abstractmethod
    async def update_conversation(self, conversation: Conversation) -> Conversation:
        """Update an existing conversation"""
        pass
    
    @abstractmethod
    async def delete_conversation(self, conversation_id: str) -> bool:
        """Delete a conversation"""
        pass
    
    @abstractmethod
    async def get_active_conversations_by_user_id(self, user_id: str, limit: int = 20) -> List[Conversation]:
        """Get active conversations for a user"""
        pass
    
    @abstractmethod
    async def add_message_to_conversation(self, conversation_id: str, message: ConversationMessage) -> ConversationMessage:
        """Add a message to a conversation"""
        pass
    
    @abstractmethod
    async def get_messages_by_conversation_id(self, conversation_id: str, limit: int = 50) -> List[ConversationMessage]:
        """Get messages for a conversation"""
        pass 