from src.domain.entities.conversation import Conversation, ConversationMessage
from src.domain.ports.repositories.conversation_repository import ConversationRepositoryPort
from src.domain.ports.repositories.charts_repository import ChartsRepositoryPort
from src.application.ports.llm_service_port import LLMServicePort
from src.application.dtos.conversation import (
    CreateConversationRequestDTO,
    SendMessageRequestDTO,
    ConversationResponseDTO,
    ConversationDetailResponseDTO,
    MessageResponseDTO
)
from src.application.dtos.LLM import LLMRequestDTO
import uuid
from datetime import datetime, timezone
from typing import List, Optional
import logging

logger = logging.getLogger(__name__)

class ChatService:
    def __init__(
        self,
        conversation_repo: ConversationRepositoryPort,
        charts_repo: ChartsRepositoryPort,
        llm_service: LLMServicePort
    ):
        self._conversation_repo = conversation_repo
        self._charts_repo = charts_repo
        self._llm_service = llm_service

    async def create_conversation(
        self, 
        user_id: str, 
        request: CreateConversationRequestDTO
    ) -> ConversationResponseDTO:
        """Create a new conversation for a chart"""
        try:
            # Verify the chart image exists and belongs to the user
            chart_image = await self._charts_repo.get_by_id(request.chart_image_id)
            if not chart_image:
                raise ValueError("Chart image not found")
            
            if chart_image.user_id != user_id:
                raise ValueError("Chart image does not belong to user")

            # Create new conversation
            conversation = Conversation(
                id=str(uuid.uuid4()),
                user_id=user_id,
                chart_image_id=request.chart_image_id,
                title=request.title,
                created_at=datetime.now(timezone.utc),
                updated_at=datetime.now(timezone.utc),
                is_active=True,
                messages=[]
            )

            # Save to repository
            await self._conversation_repo.create_conversation(conversation)

            return ConversationResponseDTO(
                id=conversation.id,
                title=conversation.title,
                created_at=conversation.created_at.isoformat(),
                updated_at=conversation.updated_at.isoformat(),
                message_count=0
            )

        except Exception as e:
            logger.error(f"Error creating conversation: {str(e)}")
            raise

    async def get_user_conversations(
        self, 
        user_id: str, 
        limit: int = 20
    ) -> List[ConversationResponseDTO]:
        """Get all conversations for a user"""
        try:
            conversations = await self._conversation_repo.get_conversations_by_user_id(
                user_id, limit
            )

            return [
                ConversationResponseDTO(
                    id=conv.id,
                    title=conv.title,
                    created_at=conv.created_at.isoformat(),
                    updated_at=conv.updated_at.isoformat(),
                    message_count=len(conv.messages)
                )
                for conv in conversations
            ]

        except Exception as e:
            logger.error(f"Error getting user conversations: {str(e)}")
            raise

    async def get_conversation_details(
        self, 
        conversation_id: str, 
        user_id: str
    ) -> ConversationDetailResponseDTO:
        """Get conversation details with all messages"""
        try:
            conversation = await self._conversation_repo.get_conversation_by_id(conversation_id)
            
            if not conversation:
                raise ValueError("Conversation not found")
            
            if conversation.user_id != user_id:
                raise ValueError("Conversation does not belong to user")

            # Convert messages to DTOs
            messages = [
                MessageResponseDTO(
                    id=msg.id,
                    message_type=msg.message_type,
                    content=msg.content,
                    timestamp=msg.timestamp.isoformat()
                )
                for msg in conversation.messages
            ]

            return ConversationDetailResponseDTO(
                id=conversation.id,
                title=conversation.title,
                created_at=conversation.created_at.isoformat(),
                updated_at=conversation.updated_at.isoformat(),
                messages=messages
            )

        except Exception as e:
            logger.error(f"Error getting conversation details: {str(e)}")
            raise

    async def send_message(
        self, 
        conversation_id: str, 
        user_id: str, 
        request: SendMessageRequestDTO
    ) -> MessageResponseDTO:
        """Send a message in a conversation and get AI response"""
        try:
            # Get conversation
            conversation = await self._conversation_repo.get_conversation_by_id(conversation_id)
            
            if not conversation:
                raise ValueError("Conversation not found")
            
            if conversation.user_id != user_id:
                raise ValueError("Conversation does not belong to user")

            # Get chart image for context
            chart_image = await self._charts_repo.get_by_id(conversation.chart_image_id)
            if not chart_image:
                raise ValueError("Chart image not found")

            # Create user message
            user_message = ConversationMessage(
                id=str(uuid.uuid4()),
                conversation_id=conversation_id,
                user_id=user_id,
                message_type="user",
                content=request.message,
                timestamp=datetime.now(timezone.utc)
            )

            # Add user message to conversation
            conversation.messages.append(user_message)
            conversation.updated_at = datetime.now(timezone.utc)

            # Build context from conversation history
            context = self._build_conversation_context(conversation.messages[:-1])  # Exclude current message
            
            # Create enhanced question with context
            enhanced_question = self._create_enhanced_question(request.message, context)

            # Get AI response using LLM service
            llm_request = LLMRequestDTO(
                image_bytes=chart_image.image_data,
                question=enhanced_question
            )

            llm_response = await self._llm_service.analyze(llm_request)

            # Create assistant message
            assistant_message = ConversationMessage(
                id=str(uuid.uuid4()),
                conversation_id=conversation_id,
                user_id=user_id,
                message_type="assistant",
                content=llm_response.answer,
                timestamp=datetime.now(timezone.utc)
            )

            # Add assistant message to conversation
            conversation.messages.append(assistant_message)
            conversation.updated_at = datetime.now(timezone.utc)

            # Save updated conversation
            await self._conversation_repo.update_conversation(conversation)

            # Return the assistant message
            return MessageResponseDTO(
                id=assistant_message.id,
                message_type=assistant_message.message_type,
                content=assistant_message.content,
                timestamp=assistant_message.timestamp.isoformat()
            )

        except Exception as e:
            logger.error(f"Error sending message: {str(e)}")
            raise

    def _build_conversation_context(self, messages: List[ConversationMessage]) -> str:
        """Build context string from conversation history"""
        if not messages:
            return ""
        
        # Take last 5 messages for context to avoid token limits
        recent_messages = messages[-5:]
        
        context_parts = []
        for i, msg in enumerate(recent_messages, 1):
            role = "User" if msg.message_type == "user" else "Assistant"
            context_parts.append(f"Turn {i} - {role}: {msg.content}")
        
        return " | ".join(context_parts)

    def _create_enhanced_question(self, current_question: str, context: str) -> str:
        """Create enhanced question with conversation context"""
        if not context:
            return current_question
        
        return f"""Previous conversation context: {context}

Current question: {current_question}

Please provide a response that takes into account the conversation history and directly addresses the current question about the chart."""

    async def delete_conversation(self, conversation_id: str, user_id: str) -> bool:
        """Delete a conversation (soft delete)"""
        try:
            conversation = await self._conversation_repo.get_conversation_by_id(conversation_id)
            
            if not conversation:
                raise ValueError("Conversation not found")
            
            if conversation.user_id != user_id:
                raise ValueError("Conversation does not belong to user")

            # Soft delete
            conversation.is_active = False
            conversation.updated_at = datetime.now(timezone.utc)
            
            await self._conversation_repo.update_conversation(conversation)
            return True

        except Exception as e:
            logger.error(f"Error deleting conversation: {str(e)}")
            raise

    async def get_conversation_statistics(self, user_id: str) -> dict:
        """Get conversation statistics for a user"""
        try:
            conversations = await self._conversation_repo.get_conversations_by_user_id(user_id, limit=1000)
            
            total_conversations = len(conversations)
            total_messages = sum(len(conv.messages) for conv in conversations)
            active_conversations = len([conv for conv in conversations if conv.is_active])
            
            return {
                "total_conversations": total_conversations,
                "active_conversations": active_conversations,
                "total_messages": total_messages,
                "average_messages_per_conversation": total_messages / total_conversations if total_conversations > 0 else 0
            }

        except Exception as e:
            logger.error(f"Error getting conversation statistics: {str(e)}")
            raise
