# ChartAnalyzer System - UML Diagrams

## 1. Use Case Diagram

```mermaid
graph TB
    subgraph "ChartAnalyzer System"
        subgraph "Actors"
            User[👤 User]
            Admin[👨‍💼 Admin]
            LLMService[🤖 LLM Service]
        end
        
        subgraph "Authentication Use Cases"
            UC1[Register Account]
            UC2[Login]
            UC3[Logout]
            UC4[Validate Token]
        end
        
        subgraph "Chart Analysis Use Cases"
            UC5[Upload Chart Image]
            UC6[Analyze Chart General]
            UC7[Ask Specific Question]
            UC8[Get Advanced Analysis]
            UC9[View Analysis History]
        end
        
        subgraph "System Management Use Cases"
            UC10[Check Server Status]
            UC11[List Available Models]
            UC12[Test Model Connection]
        end
    end
    
    %% Actor relationships
    User --> UC1
    User --> UC2
    User --> UC3
    User --> UC5
    User --> UC6
    User --> UC7
    User --> UC8
    User --> UC9
    User --> UC10
    User --> UC11
    User --> UC12
    
    Admin --> UC10
    Admin --> UC11
    Admin --> UC12
    
    %% Include relationships
    UC6 -.->|includes| UC5
    UC7 -.->|includes| UC5
    UC8 -.->|includes| UC5
    
    %% Extend relationships
    UC8 -.->|extends| UC7
    
    %% System dependencies
    UC6 -.->|uses| LLMService
    UC7 -.->|uses| LLMService
    UC8 -.->|uses| LLMService
```

## 2. System Sequence Diagrams

### 2.1 User Registration Flow

```mermaid
sequenceDiagram
    participant U as User
    participant F as Frontend
    participant API as FastAPI
    participant AS as AuthService
    participant UR as UserRepository
    participant DB as Database
    
    U->>F: Fill registration form
    F->>API: POST /auth/register
    API->>AS: register(email, password)
    AS->>AS: hash_password(password)
    AS->>UR: create_user(user)
    UR->>DB: INSERT user
    DB-->>UR: user_id
    UR-->>AS: User object
    AS-->>API: User object
    API-->>F: 201 Created
    F-->>U: Registration success
```

### 2.2 Chart Analysis Flow

```mermaid
sequenceDiagram
    participant U as User
    participant F as Frontend
    participant API as FastAPI
    participant CS as ChartService
    participant LS as LLMService
    participant CG as ChartGemma
    participant O as Ollama
    
    U->>F: Upload image + question
    F->>API: POST /charts/analyze
    API->>CS: analyze_chart(image, question)
    CS->>LS: analyze(request)
    alt Model Selection
        LS->>CG: analyze(image, question)
        CG-->>LS: analysis_result
    else Default Model
        LS->>O: analyze(image, question)
        O-->>LS: analysis_result
    end
    LS-->>CS: LLMResponse
    CS-->>API: Analysis result
    API-->>F: 200 OK
    F-->>U: Display analysis
```

### 2.3 Advanced Chart Analysis Flow

```mermaid
sequenceDiagram
    participant U as User
    participant F as Frontend
    participant API as FastAPI
    participant CS as ChartService
    participant CG as ChartGemma
    
    U->>F: Upload image + question + advanced=true
    F->>API: POST /charts/ask
    API->>CS: analyze_specialized(image, question, advanced)
    CS->>CS: Add "PROGRAM OF THOUGHT" prefix
    CS->>CG: analyze(enhanced_question)
    CG-->>CS: Advanced analysis result
    CS-->>API: Analysis result
    API-->>F: 200 OK
    F-->>U: Display advanced analysis
```

## 3. Class Diagram

```mermaid
classDiagram
    %% Domain Entities
    class User {
        +String id
        +EmailStr email
        +String password_hash
        +Boolean is_active
        +DateTime created_at
        +DateTime last_login
    }
    
    class ChartImage {
        +String id
        +String user_id
        +Bytes image_data
        +DateTime uploaded_at
    }
    
    class ChartAnalysis {
        +String id
        +String chart_image_id
        +String question
        +String answer
        +DateTime created_at
    }
    
    %% Application DTOs
    class RegisterRequestDTO {
        +String email
        +String password
    }
    
    class LoginRequestDTO {
        +String email
        +String password
    }
    
    class LLMRequestDTO {
        +Bytes image_bytes
        +String question
    }
    
    class LLMResponseDTO {
        +String answer
        +String model_used
        +Float processing_time
    }
    
    %% Application Services
    class AuthService {
        -UserRepositoryPort user_repo
        -TokenRepositoryPort token_repo
        -String secret_key
        +register(email, password) User
        +login(email, password) String
        +logout(token) void
        +validate_token(token) Boolean
    }
    
    class ChartGemmaService {
        -String gradio_url
        +analyze(request) LLMResponseDTO
        -_load_image(image_data) Image
        -_call_gradio_api(image, question) String
    }
    
    class OllamaService {
        -String host
        -String model
        +analyze(request) LLMResponseDTO
    }
    
    %% Domain Ports (Interfaces)
    class UserRepositoryPort {
        <<interface>>
        +create_user(user) void
        +get_by_email(email) User
        +get_by_id(id) User
    }
    
    class ChartsRepositoryPort {
        <<interface>>
        +save(chart_image) void
        +get_by_id(id) ChartImage
        +get_by_user_id(user_id) List~ChartImage~
    }
    
    class LLMServicePort {
        <<interface>>
        +analyze(request) LLMResponseDTO
    }
    
    %% Use Cases
    class UploadChartUseCase {
        -ChartsRepositoryPort image_repo
        +execute(image_bytes, user_id) String
    }
    
    class AnalyzeChartUseCase {
        -ChartsRepositoryPort charts_repo
        -AnalysisRepositoryPort analysis_repo
        +execute(image_bytes, question) ChartAnalysis
    }
    
    %% Relationships
    User ||--o{ ChartImage : owns
    ChartImage ||--o{ ChartAnalysis : generates
    AuthService ..|> UserRepositoryPort : uses
    ChartGemmaService ..|> LLMServicePort : implements
    OllamaService ..|> LLMServicePort : implements
    UploadChartUseCase ..|> ChartsRepositoryPort : uses
    AnalyzeChartUseCase ..|> ChartsRepositoryPort : uses
    AnalyzeChartUseCase ..|> AnalysisRepositoryPort : uses
```

## 4. Component Diagram

```mermaid
graph TB
    subgraph "Frontend Layer (React + TypeScript)"
        UI[User Interface]
        AC[Auth Context]
        CS[Chart Service]
        AS[API Service]
    end
    
    subgraph "API Gateway Layer (FastAPI)"
        Router[Router]
        Middleware[CORS Middleware]
        Auth[Authentication]
    end
    
    subgraph "Application Layer"
        UC[Use Cases]
        AServices[Application Services]
        DTOs[Data Transfer Objects]
    end
    
    subgraph "Domain Layer"
        Entities[Domain Entities]
        Ports[Domain Ports]
        Services[Domain Services]
    end
    
    subgraph "Infrastructure Layer"
        Repos[Repositories]
        External[External Services]
        DB[(PostgreSQL)]
    end
    
    subgraph "External Services"
        ChartGemma[ChartGemma Model]
        Ollama[Ollama LLM]
        Gradio[Gradio API]
    end
    
    %% Frontend connections
    UI --> AC
    UI --> CS
    CS --> AS
    
    %% API connections
    AS --> Router
    Router --> Middleware
    Router --> Auth
    
    %% Application connections
    Router --> UC
    UC --> AServices
    AServices --> DTOs
    
    %% Domain connections
    AServices --> Entities
    AServices --> Ports
    Ports --> Services
    
    %% Infrastructure connections
    Ports --> Repos
    Repos --> DB
    Services --> External
    
    %% External service connections
    External --> ChartGemma
    External --> Ollama
    External --> Gradio
```

## 5. Activity Diagram - Chart Analysis Process

```mermaid
flowchart TD
    Start([User starts analysis]) --> Upload{Upload chart image?}
    Upload -->|Yes| Validate{Image valid?}
    Validate -->|No| Error[Show error message]
    Validate -->|Yes| Question{Enter question?}
    Question -->|No| Auto[Use default analysis]
    Question -->|Yes| Model{Select model?}
    Auto --> Model
    
    Model -->|ChartGemma| CG[Process with ChartGemma]
    Model -->|Ollama| OL[Process with Ollama]
    Model -->|Default| Default[Use default model]
    
    CG --> Advanced{Advanced analysis?}
    Advanced -->|Yes| Enhance[Add Program of Thought]
    Advanced -->|No| Process[Process normally]
    Enhance --> Process
    OL --> Process
    Default --> Process
    
    Process --> Result[Generate analysis result]
    Result --> Save[Save to database]
    Save --> Display[Display result to user]
    Display --> End([Analysis complete])
    
    Error --> End
```

## 6. State Diagram - User Authentication

```mermaid
stateDiagram-v2
    [*] --> Unauthenticated
    Unauthenticated --> Registering : Register
    Unauthenticated --> LoggingIn : Login
    Registering --> Unauthenticated : Registration failed
    Registering --> Authenticated : Registration success
    LoggingIn --> Unauthenticated : Login failed
    LoggingIn --> Authenticated : Login success
    Authenticated --> Unauthenticated : Logout
    Authenticated --> TokenExpired : Token expires
    TokenExpired --> Unauthenticated : Auto logout
    TokenExpired --> Authenticated : Refresh token
```

## 7. Deployment Diagram

```mermaid
graph TB
    subgraph "Client Layer"
        Browser[Web Browser]
        Mobile[Mobile App]
    end
    
    subgraph "Frontend Server"
        React[React App<br/>Vite Dev Server]
        Static[Static Assets]
    end
    
    subgraph "Backend Server"
        FastAPI[FastAPI Application]
        Uvicorn[Uvicorn Server]
    end
    
    subgraph "Database Server"
        PostgreSQL[(PostgreSQL Database)]
        Alembic[Alembic Migrations]
    end
    
    subgraph "External Services"
        ChartGemmaAPI[ChartGemma API<br/>Gradio]
        OllamaAPI[Ollama API<br/>Local]
    end
    
    %% Connections
    Browser --> React
    Mobile --> React
    React --> Static
    React --> FastAPI
    FastAPI --> Uvicorn
    Uvicorn --> PostgreSQL
    Uvicorn --> Alembic
    FastAPI --> ChartGemmaAPI
    FastAPI --> OllamaAPI
```

## 8. Data Flow Diagram

```mermaid
graph LR
    subgraph "Input Sources"
        UserInput[User Input]
        ImageUpload[Image Upload]
        QuestionInput[Question Input]
    end
    
    subgraph "Processing"
        Auth[Authentication]
        ImageProc[Image Processing]
        LLMProc[LLM Processing]
        Analysis[Analysis Engine]
    end
    
    subgraph "Storage"
        UserDB[(User Database)]
        ImageDB[(Image Storage)]
        AnalysisDB[(Analysis History)]
    end
    
    subgraph "Output"
        Response[API Response]
        UI[User Interface]
        Logs[System Logs]
    end
    
    %% Data flows
    UserInput --> Auth
    ImageUpload --> ImageProc
    QuestionInput --> LLMProc
    
    Auth --> UserDB
    ImageProc --> ImageDB
    LLMProc --> Analysis
    Analysis --> AnalysisDB
    
    Analysis --> Response
    Response --> UI
    Auth --> Logs
    Analysis --> Logs
``` 