package org.example;

import com.google.protobuf.InvalidProtocolBufferException;
import org.ds.proto.*;

import java.util.Arrays;

public class Main {
    public static void main(String[] args) {
        System.out.println("Protocol Buffers Encoding & Decoding Demo\n");

        // 1. ترميز وفك تشفير كتاب مفرد
        encodeDecodeSingleBook();

        System.out.println("\n" + "=".repeat(50) + "\n");

        // 2. ترميز وفك تشفير مكتبة (مجموعة كتب)
        encodeDecodeLibrary();

        System.out.println("\n" + "=".repeat(50) + "\n");

        // 3. ترميز وفك تشفير طلبات الخدمة
        encodeDecodeServiceMessages();
    }

    private static void encodeDecodeSingleBook() {
        System.out.println("1. Single Book Encoding & Decoding:");

        try {
            // 🔹 الخطوة 1: إنشاء كائن Book
            Book originalBook = Book.newBuilder()
                    .setTitle("Designing Data-Intensive Applications")
                    .setAuthor(Author.newBuilder()
                            .setName("Martin Kleppmann")
                            .setCountry("United Kingdom")
                            .build())
                    .setIsbn("978-1449373320")
                    .setYear(2017)
                    .setType(BookType.TECHNICAL)
                    .addAllCategories(Arrays.asList("Databases", "Distributed Systems", "Software Architecture"))
                    .setAvailable(true)
                    .build();

            System.out.println("    Original Book Created:");
            printBookDetails(originalBook, "      ");

            // 🔹 الخطوة 2: الترميز إلى مصفوفة بايتات
            byte[] encodedData = originalBook.toByteArray();
            System.out.println("    Encoding to byte array...");
            System.out.println("       Encoded size: " + encodedData.length + " bytes");
            System.out.print("       Byte array: ");
            printByteArrayPreview(encodedData);

            // 🔹 الخطوة 3: فك التشفير من مصفوفة البايتات
            System.out.println("    Decoding from byte array...");
            Book decodedBook = Book.parseFrom(encodedData);

            System.out.println("    Decoded Book:");
            printBookDetails(decodedBook, "      ");

            // 🔹 الخطوة 4: التحقق من التكافؤ
            System.out.println("    Verification:");
            System.out.println("      Titles match: " + originalBook.getTitle().equals(decodedBook.getTitle()));
            System.out.println("      Authors match: " + originalBook.getAuthor().getName().equals(decodedBook.getAuthor().getName()));
            System.out.println("      Years match: " + (originalBook.getYear() == decodedBook.getYear()));
            System.out.println("      Full objects equal: " + originalBook.equals(decodedBook));

        } catch (InvalidProtocolBufferException e) {
            System.err.println(" Error in encoding/decoding: " + e.getMessage());
        }
    }

    private static void encodeDecodeLibrary() {
        System.out.println("2.  Library (Book Collection) Encoding & Decoding:");

        try {
            // 🔹 الخطوة 1: إنشاء مكتبة متعددة الكتب
            Book book1 = Book.newBuilder()
                    .setTitle("Clean Code: A Handbook of Agile Software Craftsmanship")
                    .setAuthor(Author.newBuilder()
                            .setName("Robert C. Martin")
                            .setCountry("USA")
                            .build())
                    .setIsbn("978-0132350884")
                    .setYear(2008)
                    .setType(BookType.TECHNICAL)
                    .addCategories("Software Engineering")
                    .addCategories("Programming")
                    .setAvailable(true)
                    .build();

            Book book2 = Book.newBuilder()
                    .setTitle("The Pragmatic Programmer: Your Journey to Mastery")
                    .setAuthor(Author.newBuilder()
                            .setName("Andrew Hunt")
                            .setCountry("USA")
                            .build())
                    .setIsbn("978-0135957059")
                    .setYear(2019)
                    .setType(BookType.TECHNICAL)
                    .addCategories("Programming")
                    .addCategories("Career Development")
                    .setAvailable(true)
                    .build();

            Book book3 = Book.newBuilder()
                    .setTitle("Domain-Driven Design: Tackling Complexity in the Heart of Software")
                    .setAuthor(Author.newBuilder()
                            .setName("Eric Evans")
                            .setCountry("USA")
                            .build())
                    .setIsbn("978-0321125217")
                    .setYear(2003)
                    .setType(BookType.TECHNICAL)
                    .addCategories("Software Design")
                    .addCategories("Architecture")
                    .setAvailable(false)
                    .build();

            Library originalLibrary = Library.newBuilder()
                    .addBooks(book1)
                    .addBooks(book2)
                    .addBooks(book3)
                    .build();

            System.out.println("    Original Library Created:");
            System.out.println("      Total books: " + originalLibrary.getBooksCount());
            for (int i = 0; i < originalLibrary.getBooksCount(); i++) {
                System.out.println("      " + (i + 1) + ". " + originalLibrary.getBooks(i).getTitle());
            }

            // 🔹 الخطوة 2: الترميز
            byte[] encodedLibrary = originalLibrary.toByteArray();
            System.out.println("    Encoding library to byte array...");
            System.out.println("       Encoded size: " + encodedLibrary.length + " bytes");
            System.out.print("       Byte array preview: ");
            printByteArrayPreview(encodedLibrary);

            // 🔹 الخطوة 3: فك التشفير
            System.out.println("    Decoding library from byte array...");
            Library decodedLibrary = Library.parseFrom(encodedLibrary);

            System.out.println("    Decoded Library:");
            System.out.println("      Total books: " + decodedLibrary.getBooksCount());
            for (int i = 0; i < decodedLibrary.getBooksCount(); i++) {
                Book book = decodedLibrary.getBooks(i);
                System.out.println("      " + (i + 1) + ". " + book.getTitle() +
                        " by " + book.getAuthor().getName() +
                        " (" + book.getYear() + ")");
            }

            // 🔹 الخطوة 4: مقارنة الكفاءة مع JSON نظرياً
            String jsonEquivalent = createJsonEquivalent(originalLibrary);
            System.out.println("    Efficiency Comparison:");
            System.out.println("      Protobuf size: " + encodedLibrary.length + " bytes");
            System.out.println("      Estimated JSON size: " + jsonEquivalent.getBytes().length + " bytes");
            int saving = ((jsonEquivalent.getBytes().length - encodedLibrary.length) * 100) / jsonEquivalent.getBytes().length;
            System.out.println("       Size reduction: " + saving + "%");

        } catch (InvalidProtocolBufferException e) {
            System.err.println(" Error in library encoding/decoding: " + e.getMessage());
        }
    }

    private static void encodeDecodeServiceMessages() {
        System.out.println("3.  Service Messages Encoding & Decoding:");

        try {
            // 🔹 ترميز وفك تشفير طلب إضافة كتاب
            Book newBook = Book.newBuilder()
                    .setTitle("gRPC: Up and Running")
                    .setAuthor(Author.newBuilder()
                            .setName("Kasun Indrasiri")
                            .setCountry("Sri Lanka")
                            .build())
                    .setIsbn("978-1492058330")
                    .setYear(2020)
                    .setType(BookType.TECHNICAL)
                    .addCategories("gRPC")
                    .addCategories("Microservices")
                    .setAvailable(true)
                    .build();

            AddBookRequest addRequest = AddBookRequest.newBuilder()
                    .setBook(newBook)
                    .build();

            System.out.println("    AddBookRequest Encoding:");
            System.out.println("      Book: " + addRequest.getBook().getTitle());

            byte[] encodedRequest = addRequest.toByteArray();
            System.out.println("       Encoded size: " + encodedRequest.length + " bytes");

            AddBookRequest decodedRequest = AddBookRequest.parseFrom(encodedRequest);
            System.out.println("       Decoded book: " + decodedRequest.getBook().getTitle());

            // 🔹 ترميز وفك تشفير استجابة إضافة كتاب
            AddBookResponse addResponse = AddBookResponse.newBuilder()
                    .setSuccess(true)
                    .setMessage("Book added successfully to library database")
                    .setBookId("LIB-2024-001")
                    .build();

            System.out.println("    AddBookResponse Encoding:");
            System.out.println("      Success: " + addResponse.getSuccess());
            System.out.println("      Message: " + addResponse.getMessage());

            byte[] encodedResponse = addResponse.toByteArray();
            System.out.println("       Encoded size: " + encodedResponse.length + " bytes");

            AddBookResponse decodedResponse = AddBookResponse.parseFrom(encodedResponse);
            System.out.println("       Decoded success: " + decodedResponse.getSuccess());
            System.out.println("       Decoded book ID: " + decodedResponse.getBookId());

            // 🔹 ترميز وفك تشفير طلب البحث
            GetBookRequest searchRequest = GetBookRequest.newBuilder()
                    .setIsbn("978-1492058330")
                    .build();

            System.out.println("    GetBookRequest Encoding:");
            System.out.println("      ISBN: " + searchRequest.getIsbn());

            byte[] encodedSearch = searchRequest.toByteArray();
            System.out.println("       Encoded size: " + encodedSearch.length + " bytes");
            System.out.print("       Byte array: ");
            printByteArrayPreview(encodedSearch);

            GetBookRequest decodedSearch = GetBookRequest.parseFrom(encodedSearch);
            System.out.println("       Decoded ISBN: " + decodedSearch.getIsbn());

        } catch (InvalidProtocolBufferException e) {
            System.err.println(" Error in service messages encoding/decoding: " + e.getMessage());
        }
    }

    // ========== Methods ==========

    private static void printBookDetails(Book book, String indent) {
        System.out.println(indent + "Title: " + book.getTitle());
        System.out.println(indent + "Author: " + book.getAuthor().getName() + " (" + book.getAuthor().getCountry() + ")");
        System.out.println(indent + "ISBN: " + book.getIsbn());
        System.out.println(indent + "Year: " + book.getYear());
        System.out.println(indent + "Type: " + book.getType());
        System.out.println(indent + "Categories: " + book.getCategoriesList());
        System.out.println(indent + "Available: " + book.getAvailable());
    }

    private static void printByteArrayPreview(byte[] data) {
        int previewLength = Math.min(data.length, 15);
        for (int i = 0; i < previewLength; i++) {
            System.out.print(data[i] + " ");
        }
        if (data.length > previewLength) {
            System.out.print("... (+" + (data.length - previewLength) + " more bytes)");
        }
        System.out.println();
    }

    private static String createJsonEquivalent(Library library) {
        // محاكاة تقريبية لحجم JSON
        StringBuilder json = new StringBuilder("[");
        for (Book book : library.getBooksList()) {
            if (json.length() > 1) json.append(",");
            json.append("{")
                    .append("\"title\":\"").append(book.getTitle()).append("\",")
                    .append("\"author\":{")
                    .append("\"name\":\"").append(book.getAuthor().getName()).append("\",")
                    .append("\"country\":\"").append(book.getAuthor().getCountry()).append("\"")
                    .append("},")
                    .append("\"isbn\":\"").append(book.getIsbn()).append("\",")
                    .append("\"year\":").append(book.getYear()).append(",")
                    .append("\"type\":\"").append(book.getType()).append("\",")
                    .append("\"categories\":").append(book.getCategoriesList().toString()).append(",")
                    .append("\"available\":").append(book.getAvailable())
                    .append("}");
        }
        json.append("]");
        return json.toString();
    }
}