import model.SequencedLine;
import consumer.EncryptionProcessor;
import consumer.EncryptionConsumer;
import producer.FileProducer;
import writer.FileWriterConsumer;

import java.util.Scanner;
import java.util.concurrent.PriorityBlockingQueue;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.Executors;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.atomic.AtomicBoolean;
import java.util.concurrent.atomic.AtomicInteger;

public class Main {
    public static void main(String[] args) {
        System.out.println("=== Parallel Encryption System ===");

        Scanner scanner = new Scanner(System.in);

        // Basic configuration
        System.out.print("Input file (default: input.txt): ");
        String inputFile = scanner.nextLine().trim();
        if (inputFile.isEmpty()) inputFile = "input.txt";

        System.out.print("Output file (default: output.txt): ");
        String outputFile = scanner.nextLine().trim();
        if (outputFile.isEmpty()) outputFile = "output.txt";

        System.out.print("Number of consumers (default: 3): ");
        int numConsumers = 3;
        try {
            String input = scanner.nextLine().trim();
            if (!input.isEmpty()) numConsumers = Integer.parseInt(input);
        } catch (NumberFormatException e) {
            System.out.println("Invalid input. Using 3 consumers.");
        }

        System.out.print("Encryption key (default: 3): ");
        int encryptionKey = 3;
        try {
            String input = scanner.nextLine().trim();
            if (!input.isEmpty()) encryptionKey = Integer.parseInt(input);
        } catch (NumberFormatException e) {
            System.out.println("Invalid input. Using key 3.");
        }

        scanner.close();

        // Create queues and shared objects
        PriorityBlockingQueue<SequencedLine> inputQueue = new PriorityBlockingQueue<>();
        PriorityBlockingQueue<SequencedLine> outputQueue = new PriorityBlockingQueue<>();
        AtomicBoolean productionFinished = new AtomicBoolean(false);
        AtomicInteger activeConsumers = new AtomicInteger(0);

        // Create processor
        EncryptionProcessor processor = new EncryptionProcessor(encryptionKey);

        System.out.println("\nStarting encryption...");

        // Start all components
        ExecutorService consumerExecutor = Executors.newFixedThreadPool(numConsumers);

        // Start producer
        Thread producerThread = new Thread(new FileProducer(inputFile, inputQueue, productionFinished));
        producerThread.start();

        // Start consumers
        for (int i = 1; i <= numConsumers; i++) {
            EncryptionConsumer consumer = new EncryptionConsumer(i, inputQueue, outputQueue,
                    productionFinished, activeConsumers, processor);
            consumerExecutor.submit(consumer);
        }

        // Start writer
        Thread writerThread = new Thread(new FileWriterConsumer(outputFile, outputQueue, activeConsumers));
        writerThread.start();

        // Wait for completion
        try {
            // Wait for producer to finish
            producerThread.join();

            // Shutdown consumers and wait
            consumerExecutor.shutdown();
            consumerExecutor.awaitTermination(1, TimeUnit.MINUTES);

            // Wait for writer to finish
            writerThread.join();

            System.out.println("Encryption completed successfully!");

        } catch (InterruptedException e) {
            System.out.println("Process was interrupted");
            consumerExecutor.shutdownNow();
            Thread.currentThread().interrupt();
        }
    }
}