package writer;

import model.SequencedLine;
import java.io.BufferedWriter;
import java.io.FileWriter;
import java.io.IOException;
import java.util.concurrent.PriorityBlockingQueue;
import java.util.concurrent.atomic.AtomicInteger;


public class FileWriterConsumer implements Runnable {
    private final String outputFilePath;
    private final PriorityBlockingQueue<SequencedLine> outputQueue;
    private final AtomicInteger activeConsumers;

    public FileWriterConsumer(String outputFilePath,
                              PriorityBlockingQueue<SequencedLine> outputQueue,
                              AtomicInteger activeConsumers) {
        this.outputFilePath = outputFilePath;
        this.outputQueue = outputQueue;
        this.activeConsumers = activeConsumers;
    }

    @Override
    public void run() {
        System.out.println(" Writer started: " + outputFilePath);
        int writtenCount = 0;

        try (BufferedWriter writer = new BufferedWriter(new FileWriter(outputFilePath))) {

            // Continue writing until all consumers are done and queue is empty
            while (activeConsumers.get() > 0 || !outputQueue.isEmpty()) {
                try {
                    SequencedLine line = outputQueue.take();

                    // Write the line to the output file
                    writer.write(line.getLine());
                    writer.newLine();
                    writtenCount++;

                    // Progress reporting
                    if (writtenCount % 25 == 0) {
                        System.out.println(" Writer: " + writtenCount + " lines written (last sequence: " +
                                line.getSequenceNumber() + ")");
                    }

                } catch (InterruptedException e) {
                    System.err.println(" Writer was interrupted while waiting for data");
                    Thread.currentThread().interrupt();
                    break;
                }
            }

            System.out.println(" Writer finished: " + writtenCount + " lines written in perfect order");

        } catch (IOException e) {
            System.err.println(" File write error: " + e.getMessage());
            e.printStackTrace();
        }

        // Final status report
        System.out.println(" Writer completed. Final queue size: " + outputQueue.size());
    }

    public String getOutputFilePath() {
        return outputFilePath;
    }

    public int getRemainingQueueSize() {
        return outputQueue.size();
    }
}