package producer;

import model.SequencedLine;
import java.io.BufferedReader;
import java.io.FileReader;
import java.io.IOException;
import java.util.concurrent.PriorityBlockingQueue;
import java.util.concurrent.atomic.AtomicBoolean;

public class FileProducer implements Runnable {
    private final String inputFilePath;
    private final PriorityBlockingQueue<SequencedLine> inputQueue;
    private final AtomicBoolean productionFinished;
    private int sequenceCounter = 0;

    public FileProducer(String inputFilePath,
                        PriorityBlockingQueue<SequencedLine> inputQueue,
                        AtomicBoolean productionFinished) {
        this.inputFilePath = inputFilePath;
        this.inputQueue = inputQueue;
        this.productionFinished = productionFinished;
    }

    @Override
    public void run() {
        System.out.println("Producer started reading file: " + inputFilePath);
        int lineCount = 0;

        try (BufferedReader reader = new BufferedReader(new FileReader(inputFilePath))) {
            String line;

            // Read file line by line
            while ((line = reader.readLine()) != null) {
                if (!line.trim().isEmpty()) {
                    // Put line in queue - will wait if queue is full
                    SequencedLine sequencedLine = new SequencedLine(sequenceCounter, line);
                    sequenceCounter++; // Prepare for next line
                    inputQueue.put(sequencedLine);
                    lineCount++;

                    // Progress reporting every 50 lines
                    if (lineCount % 50 == 0) {
                        System.out.println("Producer read " + lineCount + " lines...");
                    }
                }
            }

            System.out.println("Producer finished: " + lineCount + " lines read from file");

        } catch (IOException e) {
            System.err.println("Error reading file: " + e.getMessage());
            e.printStackTrace();
        }  finally {
            // Signal that production is finished
            productionFinished.set(true);
            System.out.println("Producer: File reading completed");
        }
    }

    public String getInputFilePath() {
        return inputFilePath;
    }
}