package producer;

import java.io.BufferedReader;
import java.io.FileReader;
import java.io.IOException;
import java.util.concurrent.BlockingQueue;
import java.util.concurrent.atomic.AtomicBoolean;

public class FileProducer implements Runnable {
    private final String inputFilePath;
    private final BlockingQueue<String> inputQueue;
    private final AtomicBoolean productionFinished;

    public FileProducer(String inputFilePath,
                        BlockingQueue<String> inputQueue,
                        AtomicBoolean productionFinished) {
        this.inputFilePath = inputFilePath;
        this.inputQueue = inputQueue;
        this.productionFinished = productionFinished;
    }

    @Override
    public void run() {
        System.out.println("Producer started reading file: " + inputFilePath);
        int lineCount = 0;

        try (BufferedReader reader = new BufferedReader(new FileReader(inputFilePath))) {
            String line;

            // Read file line by line
            while ((line = reader.readLine()) != null) {
                if (!line.trim().isEmpty()) {
                    // Put line in queue - will wait if queue is full
                    inputQueue.put(line);
                    lineCount++;

                    // Progress reporting every 50 lines
                    if (lineCount % 50 == 0) {
                        System.out.println("Producer read " + lineCount + " lines...");
                    }
                }
            }

            System.out.println("Producer finished: " + lineCount + " lines read from file");

        } catch (IOException e) {
            System.err.println("Error reading file: " + e.getMessage());
            e.printStackTrace();
        } catch (InterruptedException e) {
            System.err.println("Producer was interrupted while waiting for queue space");
            Thread.currentThread().interrupt(); // Restore interrupt status
        } finally {
            // Signal that production is finished
            productionFinished.set(true);
            System.out.println("Producer: File reading completed");
        }
    }

    public String getInputFilePath() {
        return inputFilePath;
    }
}