import org.apache.zookeeper.*;
import org.apache.zookeeper.data.Stat;

import java.io.IOException;
import java.util.Collections;
import java.util.List;

public class LeaderElection implements Watcher {
    private static final String address = "192.168.97.15:2181";
    private static final int SESSION_TIMEOUT = 3000; //dead client
    private static final String ELECTION_NAMESPACE = "/election";

    private String currentZnodeName;
    private ZooKeeper zooKeeper;
    private boolean isLeader = false;


    public static void main(String[] args) throws IOException, InterruptedException, KeeperException {
        LeaderElection leaderElection = new LeaderElection();
        leaderElection.connectToZookeeper();
        leaderElection.volunteerForLeadership();
        leaderElection.electLeaderWithSmartWatching();
        leaderElection.run();
        leaderElection.close();
        System.out.println("Successfully Closed");
    }

    public void volunteerForLeadership() throws InterruptedException, KeeperException {
        String znodePrefix = ELECTION_NAMESPACE + "/c_";
        String znodeFullPath = zooKeeper.create(znodePrefix, new byte[]{}, ZooDefs.Ids.OPEN_ACL_UNSAFE, CreateMode.EPHEMERAL_SEQUENTIAL);

        System.out.println(znodeFullPath);
        this.currentZnodeName = znodeFullPath.replace(ELECTION_NAMESPACE + "/", "");
    }

    public void electLeaderWithSmartWatching() throws InterruptedException, KeeperException {
        List<String> children = zooKeeper.getChildren(ELECTION_NAMESPACE, false);
        Collections.sort(children);

        String smallestChild = children.get(0);

        if (smallestChild.equals(currentZnodeName)) {
            isLeader = true;
            System.out.println("I'm the LEADER: " + currentZnodeName);
        } else {
            isLeader = false;
            int currentIndex = children.indexOf(currentZnodeName);
            String previousNode = children.get(currentIndex - 1);

            System.out.println("I'm a FOLLOWER. Watching previous node: " + previousNode);
            watchPreviousNode(previousNode);
        }
    }

    private void watchPreviousNode(String previousNode) throws KeeperException, InterruptedException {
        String previousNodePath = ELECTION_NAMESPACE + "/" + previousNode;
        zooKeeper.exists(previousNodePath, this);
    }

    private void close() throws InterruptedException {
        this.zooKeeper.close();
    }

    public void connectToZookeeper() throws IOException {
        this.zooKeeper = new ZooKeeper(address, SESSION_TIMEOUT, this);
    }

    public void run() throws InterruptedException {
        synchronized (zooKeeper) {
            zooKeeper.wait();
        }
    }

    @Override
    public void process(WatchedEvent watchedEvent) {
        switch (watchedEvent.getType()) {
            case None:
                if (watchedEvent.getState() == Event.KeeperState.SyncConnected) {
                    System.out.println("Successfully connected to Zookeeper");
                } else if (watchedEvent.getState() == Event.KeeperState.Disconnected) {
                    synchronized (zooKeeper) {
                        System.out.println("Disconnected from Zookeeper");
                        zooKeeper.notifyAll();
                    }
                } else if (watchedEvent.getState() == Event.KeeperState.Closed) {
                    System.out.println("Closed Successfully");
                }
            case NodeDeleted:
                System.out.println("Previous node deleted - re-electing leader");
                try {
                    electLeaderWithSmartWatching();
                } catch (Exception e) {
                    e.printStackTrace();
                }
                break;

        }
    }
}
