package com.distributed.search.cluster;

import com.distributed.search.grpc.SearchClient;
import com.distributed.search.grpc.SearchServiceImpl;
import com.distributed.search.web.CoordinatorWebServer; // New Import
import io.grpc.Server;
import io.grpc.ServerBuilder;
import org.apache.zookeeper.KeeperException;

import java.io.IOException;
import java.net.InetAddress;
import java.net.UnknownHostException;

/**
 * Defines the actions taken when a node's role changes in the cluster.
 */
public class OnElectionAction implements OnElectionCallback {

    private final ServiceRegistry serviceRegistry;
    private final LeaderHttpRegistry leaderHttpRegistry; // For discovery
    private final SearchClient searchClient; // To perform search
    private final int port;
    private Server grpcServer; // The gRPC server instance for Workers
    private CoordinatorWebServer leaderInternalWebServer; // The Internal API
    private final String sharedDirectoryPath = "storage"; // The central storage folder

    public OnElectionAction(ServiceRegistry serviceRegistry,
                            LeaderHttpRegistry leaderHttpRegistry,
                            SearchClient searchClient,
                            int port) {
        this.serviceRegistry = serviceRegistry;
        this.leaderHttpRegistry = leaderHttpRegistry;
        this.searchClient = searchClient;
        this.port = port;
    }

    /**
     * Called when this node is elected as the Coordinator (Leader).
     */
    @Override
    public void onElectedToBeLeader() {
        serviceRegistry.unregisterFromCluster();
        serviceRegistry.registerForUpdates();

        // 1. Start the Internal HTTP API (Running on port + 1000)
        int internalApiPort = port + 1000;
        leaderInternalWebServer = new CoordinatorWebServer(internalApiPort, searchClient, sharedDirectoryPath, serviceRegistry);
        leaderInternalWebServer.startServer();

        // 2. Register the Internal API address in Zookeeper for the Frontend to see
        try {
            String leaderApiAddress = String.format("http://localhost:%d/search", internalApiPort);
            leaderHttpRegistry.registerLeader(leaderApiAddress);
        } catch (KeeperException | InterruptedException e) {
            e.printStackTrace();
        }

        System.out.println("Node promoted to Leader. Internal API is live.");

        // Stop gRPC server if it was acting as a worker
        if (grpcServer != null) {
            grpcServer.shutdown();
        }
    }

    /**
     * Called when this node is a Worker.
     */
    @Override
    public void onWorker() {
        try {
            boolean started = startGrpcServer();

            if (started) {
                String currentServerAddress = "localhost:" + port;
                serviceRegistry.registerToCluster(currentServerAddress);
                System.out.println("Worker registered successfully on: " + currentServerAddress);
            } else {
                System.err.println("Worker failed to start gRPC server. Registration aborted.");
            }
        } catch (Exception e) {
            e.printStackTrace();
        }
    }

    /**
     * Helper method to start the gRPC server that runs SearchServiceImpl logic.
     */
    private boolean startGrpcServer() {
        try {
            grpcServer = ServerBuilder.forPort(port)
                    .addService(new SearchServiceImpl(sharedDirectoryPath))
                    .build();
            grpcServer.start();
            return true;
        } catch (IOException e) {
            System.err.println("gRPC Server Error: " + e.getMessage());
            return false;
        }
    }
}