package com.distributed.search.grpc;

import com.distributed.search.model.*;
import com.distributed.search.logic.*;
import io.grpc.stub.StreamObserver;
import java.nio.file.Path;
import java.nio.file.Paths;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

/**
 * Implements gRPC Search Service on the Worker nodes.
 */

public class SearchServiceImpl extends SearchServiceGrpc.SearchServiceImplBase {

    private final String sharedDirectoryPath;

    public SearchServiceImpl(String sharedDirectoryPath) {
        this.sharedDirectoryPath = sharedDirectoryPath;
    }
    /**
     * Local Document Frequency Statistics.
     */
    @Override
    public void getDocumentStats(StatRequest request, StreamObserver<StatResponse> responseObserver) {
        List<String> terms = request.getTermsList();
        int startIndex = request.getStartIndex();
        int count = request.getCount();

        List<String> allFiles = FileManager.getSortedDocumentNames(sharedDirectoryPath);
        int endIndex = Math.min(startIndex + count, allFiles.size());
        List<String> assignedFiles = allFiles.subList(startIndex, endIndex);

        Map<String, Integer> termToDocCount = new HashMap<>();

        for (String term : terms) {
            int docsWithTerm = 0;
            for (String docName : assignedFiles) {
                Path filePath = Paths.get(sharedDirectoryPath, docName);
                List<String> words = TFIDFCalculator.getWordsFromDocument(filePath);

                if (words.contains(term.toLowerCase())) {
                    docsWithTerm++;
                }
            }
            termToDocCount.put(term, docsWithTerm);
        }

        StatResponse response = StatResponse.newBuilder()
                .putAllTermToDocumentCount(termToDocCount)
                .build();

        responseObserver.onNext(response);
        responseObserver.onCompleted();
    }
    /**
     * Final Scoring using Global IDF received from Coordinator.
     */
    @Override
    public void getFinalScores(CalculationRequest request, StreamObserver<SearchResponse> responseObserver) {
        List<String> terms = request.getTermsList();
        Map<String, Double> globalIdfs = request.getGlobalIdfsMap();
        int startIndex = request.getStartIndex();
        int count = request.getCount();

        List<String> allFiles = FileManager.getSortedDocumentNames(sharedDirectoryPath);
        int endIndex = Math.min(startIndex + count, allFiles.size());
        List<String> assignedFiles = allFiles.subList(startIndex, endIndex);

        SearchResponse.Builder responseBuilder = SearchResponse.newBuilder();

        for (String docName : assignedFiles) {
            double docScore = 0.0;

            Path filePath = Paths.get(sharedDirectoryPath, docName);
            List<String> words = TFIDFCalculator.getWordsFromDocument(filePath);

            if (!words.isEmpty()) {
                for (String term : terms) {
                    double tf = TFIDFCalculator.calculateTermFrequency(words, term);
                    double idf = globalIdfs.getOrDefault(term, 0.0);
                    docScore += (tf * idf);
                }
            }

            responseBuilder.addResults(SearchResponse.DocumentResult.newBuilder()
                    .setDocumentName(docName)
                    .setScore(docScore)
                    .build());
        }

        responseObserver.onNext(responseBuilder.build());
        responseObserver.onCompleted();
    }
}