package com.distributed.search.web;

import com.distributed.search.grpc.SearchClient;
import com.distributed.search.logic.FileManager;
import com.distributed.search.model.SearchResponse;
import com.google.gson.Gson;
import com.sun.net.httpserver.HttpExchange;
import com.sun.net.httpserver.HttpServer;
import com.distributed.search.cluster.ServiceRegistry;

import java.io.IOException;
import java.io.OutputStream;
import java.net.InetSocketAddress;
import java.util.Arrays;
import java.util.List;
import java.util.concurrent.Executors;
import java.net.URLDecoder;
import java.nio.charset.StandardCharsets;
/**
 * Internal HTTP Server running on the Leader node.
 * It receives search queries from the Frontend and returns results in JSON format.
 */
public class CoordinatorWebServer {
    private static final String SEARCH_ENDPOINT = "/search";
    private final int port;
    private final SearchClient searchClient;
    private final String storageDir;
    private final ServiceRegistry serviceRegistry;
    private HttpServer server;
    private final Gson gson = new Gson();

    public CoordinatorWebServer(int port, SearchClient searchClient, String storageDir, ServiceRegistry serviceRegistry) {
        this.port = port;
        this.searchClient = searchClient;
        this.storageDir = storageDir;
        this.serviceRegistry = serviceRegistry;
    }

    public void startServer() {
        try {
            this.server = HttpServer.create(new InetSocketAddress(port), 0);
            server.createContext(SEARCH_ENDPOINT, this::handleSearchRequest);
            server.setExecutor(Executors.newFixedThreadPool(8));
            server.start();
            System.out.println("Leader Internal Web Server started on port: " + port);
        } catch (IOException e) {
            e.printStackTrace();
        }
    }

    public void stopServer() {
        if (server != null) server.stop(0);
    }

    /**
     * Handles search requests from the Frontend.
     * Expects query param: /search?query=word1+word2
     */
    private void handleSearchRequest(HttpExchange exchange) throws IOException {
        try {
            String query = exchange.getRequestURI().getQuery();
            if (query == null || !query.contains("=")) {
                sendResponse("{}".getBytes(), exchange, 400);
                return;
            }

            String queryValue = query.substring(query.indexOf("=") + 1);

            List<String> terms = Arrays.asList(queryValue.toLowerCase().split("[\\s+]+"));
            System.out.println("Leader is searching for terms: " + terms);

            List<String> allFiles = FileManager.getSortedDocumentNames(storageDir);
            List<String> workers = serviceRegistry.getAllServiceAddresses();
            searchClient.updateWorkers(workers);

            List<SearchResponse.DocumentResult> results = searchClient.performSearch(terms, allFiles);

            String jsonResponse = gson.toJson(results);
            sendResponse(jsonResponse.getBytes(), exchange, 200);

        } catch (Exception e) {
            e.printStackTrace();
            sendResponse("Internal Server Error".getBytes(), exchange, 500);
        }
    }

    private void sendResponse(byte[] responseBytes, HttpExchange exchange, int statusCode) throws IOException {
        exchange.getResponseHeaders().add("Content-Type", "application/json");
        exchange.sendResponseHeaders(statusCode, responseBytes.length);
        try (OutputStream os = exchange.getResponseBody()) {
            os.write(responseBytes);
            os.flush();
        }
    }
}