package com.distributed.search.cluster;

import org.apache.zookeeper.*;
import org.apache.zookeeper.data.Stat;
import java.util.List;

/**
 * This class acts as a shared registry in Zookeeper where the elected Leader
 * publishes its HTTP server address so the Frontend can discover it.
 */
public class LeaderHttpRegistry implements Watcher {
    private static final String LEADER_HTTP_ZNODE = "/leader_http_registry";
    private final ZooKeeper zooKeeper;
    private String leaderAddress = null;

    public LeaderHttpRegistry(ZooKeeper zooKeeper) {
        this.zooKeeper = zooKeeper;
        createRegistryZnode();
    }

    /**
     * Ensures that the parent Znode for the registry exists.
     */
    private void createRegistryZnode() {
        try {
            if (zooKeeper.exists(LEADER_HTTP_ZNODE, false) == null) {
                zooKeeper.create(LEADER_HTTP_ZNODE, new byte[]{}, ZooDefs.Ids.OPEN_ACL_UNSAFE, CreateMode.PERSISTENT);
            }
        } catch (KeeperException | InterruptedException e) {
            e.printStackTrace();
        }
    }

    /**
     * Leader uses this method to register its HTTP address (IP:Port).
     * We use an EPHEMERAL node so that if the leader crashes, the registry is cleared.
     */
    public void registerLeader(String address) throws KeeperException, InterruptedException {
        String leaderPath = LEADER_HTTP_ZNODE + "/address";

        // If an old address exists, delete it first
        if (zooKeeper.exists(leaderPath, false) != null) {
            zooKeeper.delete(leaderPath, -1);
        }

        zooKeeper.create(leaderPath, address.getBytes(),
                ZooDefs.Ids.OPEN_ACL_UNSAFE, CreateMode.EPHEMERAL);

        System.out.println("Leader HTTP address registered: " + address);
    }

    /**
     * Frontend uses this method to get the current leader's HTTP address.
     * It also sets a watcher to stay updated if the leader changes.
     */
    public synchronized String getLeaderAddress() {
        try {
            updateLeaderAddress();
        } catch (KeeperException | InterruptedException e) {
            e.printStackTrace();
        }
        return leaderAddress;
    }

    private synchronized void updateLeaderAddress() throws KeeperException, InterruptedException {
        String leaderPath = LEADER_HTTP_ZNODE + "/address";
        Stat stat = zooKeeper.exists(leaderPath, this);

        if (stat != null) {
            byte[] data = zooKeeper.getData(leaderPath, this, stat);
            this.leaderAddress = new String(data);
        } else {
            this.leaderAddress = null;
        }
    }

    /**
     * Watcher to handle events if the leader node is created, deleted, or data changes.
     */
    @Override
    public void process(WatchedEvent event) {
        try {
            updateLeaderAddress();
            System.out.println("Leader registry updated. Current leader: " + leaderAddress);
        } catch (KeeperException | InterruptedException e) {
            e.printStackTrace();
        }
    }
}