package com.distributed.search.cluster;

import com.distributed.search.grpc.SearchServiceImpl;
import io.grpc.Server;
import io.grpc.ServerBuilder;

import java.io.IOException;

/**
 * Defines the actions taken when a node's role changes in the cluster.
 */
public class OnElectionAction implements OnElectionCallback {

    private final ServiceRegistry serviceRegistry;
    private final int port;
    private Server grpcServer; // The gRPC server instance for Workers
    private final String sharedDirectoryPath = "storage"; // The central storage folder

    public OnElectionAction(ServiceRegistry serviceRegistry, int port) {
        this.serviceRegistry = serviceRegistry;
        this.port = port;
    }

    /**
     * Called when this node is elected as the Coordinator (Leader).
     */
    @Override
    public void onElectedToBeLeader() {
        // If it was a worker, unregister itself from the worker list
        serviceRegistry.unregisterFromCluster();

        // Start watching for other active workers in the cluster
        serviceRegistry.registerForUpdates();

        System.out.println("Node elected as Leader. Monitoring the cluster...");

        // Note: The Coordinator doesn't need to run a gRPC server for searching.
        // It acts as a client that sends requests to workers.
        if (grpcServer != null) {
            grpcServer.shutdown();
        }
    }

    /**
     * Called when this node is a Worker.
     */
    @Override
    public void onWorker() {
        try {
            // نغير الترتيب: نشغل السيرفر أولاً
            boolean started = startGrpcServer();

            if (started) {
                String currentServerAddress = "localhost:" + port;
                serviceRegistry.registerToCluster(currentServerAddress);
                System.out.println("Worker registered successfully on: " + currentServerAddress);
            } else {
                System.err.println("Worker failed to start gRPC server. Registration aborted.");
            }
        } catch (Exception e) {
            e.printStackTrace();
        }
    }

    /**
     * Helper method to start the gRPC server that runs SearchServiceImpl logic.
     */
    private boolean startGrpcServer() {
        try {
            grpcServer = ServerBuilder.forPort(port)
                    .addService(new SearchServiceImpl(sharedDirectoryPath))
                    .build();
            grpcServer.start();
            return true;
        } catch (IOException e) {
            System.err.println("gRPC Server Error: " + e.getMessage());
            return false;
        }
    }
}