package com.distributed.search;

import com.distributed.search.cluster.LeaderElection;
import com.distributed.search.cluster.OnElectionAction;
import com.distributed.search.cluster.ServiceRegistry;
import com.distributed.search.grpc.SearchClient;
import com.distributed.search.logic.FileManager;
import org.apache.zookeeper.KeeperException;
import org.apache.zookeeper.WatchedEvent;
import org.apache.zookeeper.Watcher;
import org.apache.zookeeper.ZooKeeper;

import java.io.IOException;
import java.util.Arrays;
import java.util.List;
import java.util.Scanner;

/**
 * Main Entry point for the Distributed Search Engine.
 */
public class Application implements Watcher {
    private static final String ZOOKEEPER_ADDRESS = "192.168.96.198:2181"; // Change if ZK is on another machine
    private static final int SESSION_TIMEOUT = 3000;
    private static final int DEFAULT_PORT = 8080;
    private static final String STORAGE_DIR = "storage"; // Folder containing .txt files

    private ZooKeeper zooKeeper;
    private static boolean isLeader = false;

    public static void main(String[] args) throws IOException, InterruptedException, KeeperException {
        // 1. Force the system to use IPv4 and disable native transports that cause 'unix' errors
        System.setProperty("io.grpc.netty.shaded.io.netty.transport.noNative", "true");
        System.setProperty("java.net.preferIPv4Stack", "true");

        // Use port from arguments to allow multiple instances on the same machine
        int currentServerPort = args.length == 1 ? Integer.parseInt(args[0]) : DEFAULT_PORT;

        Application application = new Application();
        ZooKeeper zooKeeper = application.connectToZookeeper();

        ServiceRegistry serviceRegistry = new ServiceRegistry(zooKeeper);
        SearchClient searchClient = new SearchClient();

        // Action to take when election completes
        OnElectionAction onElectionAction = new OnElectionAction(serviceRegistry, currentServerPort) {
            @Override
            public void onElectedToBeLeader() {
                super.onElectedToBeLeader();
                isLeader = true; // Mark this node as the leader
            }
        };

        LeaderElection leaderElection = new LeaderElection(zooKeeper, onElectionAction);
        leaderElection.volunteerForLeadership();
        leaderElection.reelectLeader();

        // If this node is the Leader, enter the Search Loop
        application.enterSearchLoop(serviceRegistry, searchClient);

        application.close();
    }

    /**
     * The Main UI Loop: Only active for the Leader/Coordinator node.
     */
    private void enterSearchLoop(ServiceRegistry serviceRegistry, SearchClient searchClient) throws InterruptedException {
        Scanner scanner = new Scanner(System.in);

        while (true) {
            if (isLeader) {
                System.out.println("\n[Coordinator] Enter search query (or 'exit' to quit):");
                String input = scanner.nextLine();

                if (input.equalsIgnoreCase("exit")) break;
                if (input.trim().isEmpty()) continue;

                // 1. Get current active workers from Zookeeper
                List<String> workers = serviceRegistry.getAllServiceAddresses();
                if (workers.isEmpty()) {
                    System.out.println("No workers registered yet. Please wait...");
                    continue;
                }

                // 2. Prepare search data
                List<String> terms = Arrays.asList(input.toLowerCase().split("\\s+"));
                List<String> allFiles = FileManager.getSortedDocumentNames(STORAGE_DIR);

                if (allFiles.isEmpty()) {
                    System.out.println("No documents found in 'storage' directory.");
                    continue;
                }

                // 3. Update gRPC channels and perform distributed search
                System.out.println("Searching in " + allFiles.size() + " files across " + workers.size() + " workers...");
                searchClient.updateWorkers(workers);
                searchClient.performSearch(terms, allFiles);

            } else {
                // If Worker: Just wait and keep the connection alive
                synchronized (zooKeeper) {
                    zooKeeper.wait(5000);
                }
            }
        }
    }

    public ZooKeeper connectToZookeeper() throws IOException {
        this.zooKeeper = new ZooKeeper(ZOOKEEPER_ADDRESS, SESSION_TIMEOUT, this);
        return zooKeeper;
    }

    private void close() throws InterruptedException {
        this.zooKeeper.close();
    }

    @Override
    public void process(WatchedEvent event) {
        if (event.getType() == Event.EventType.None && event.getState() == Event.KeeperState.SyncConnected) {
            System.out.println("Successfully connected to Zookeeper");
        }
    }
}