package com.distributed.search.grpc;
import java.net.InetSocketAddress;

import com.distributed.search.model.*;
import io.grpc.ManagedChannel;
// انتبه: هذا الـ Import المظلل هو الذي يحل مشكلة الـ NameResolver في الـ Fat JAR
import io.grpc.netty.shaded.io.grpc.netty.NettyChannelBuilder;
import java.util.*;

public class SearchClient {
    private final Map<String, SearchServiceGrpc.SearchServiceBlockingStub> stubs = new HashMap<>();
    private final List<ManagedChannel> channels = new ArrayList<>();

    public void updateWorkers(List<String> workerAddresses) {
        // إغلاق القنوات القديمة
        for (ManagedChannel channel : channels) {
            channel.shutdownNow();
        }
        channels.clear();
        stubs.clear();

        for (String address : workerAddresses) {
            try {
                // تقسيم العنوان يدويًا
                String[] parts = address.split(":");
                String host = parts[0];
                int port = Integer.parseInt(parts[1]);

                // استخدام NettyChannelBuilder.forAddress حصراً
                // هذا يتجاوز نظام الـ NameResolver تماماً ويمنع خطأ الـ 'unix'
                ManagedChannel channel = NettyChannelBuilder.forAddress(new InetSocketAddress(host, port))
                        .usePlaintext()
                        .build();

                channels.add(channel);
                stubs.put(address, SearchServiceGrpc.newBlockingStub(channel));
                System.out.println("Successfully connected to Worker: " + host + ":" + port);

            } catch (Exception e) {
                System.err.println("Failed to connect to " + address + ": " + e.getMessage());
                e.printStackTrace(); // مهم: سيظهر مكان رمي الاستثناء بالضبط
            }
        }
    }

    public void performSearch(List<String> terms, List<String> allFiles) {
        if (stubs.isEmpty()) {
            System.out.println("No workers available.");
            return;
        }

        // --- Phase 1: Global Stats ---
        Map<String, Integer> globalTermCounts = new HashMap<>();
        int filesPerWorker = (int) Math.ceil((double) allFiles.size() / stubs.size());
        int currentFileIndex = 0;
        List<String> workerList = new ArrayList<>(stubs.keySet());

        for (String address : workerList) {
            int count = Math.min(filesPerWorker, allFiles.size() - currentFileIndex);
            if (count <= 0) break;

            StatRequest request = StatRequest.newBuilder()
                    .addAllTerms(terms)
                    .setStartIndex(currentFileIndex)
                    .setCount(count)
                    .build();
            try {
                StatResponse response = stubs.get(address).getDocumentStats(request);
                response.getTermToDocumentCountMap().forEach((term, docCount) ->
                        globalTermCounts.merge(term, docCount, Integer::sum));
            } catch (Exception e) { System.err.println("Worker " + address + " Phase 1 error"); }
            currentFileIndex += count;
        }

        // --- Calculate Global IDF ---
        Map<String, Double> globalIdfs = new HashMap<>();
        for (String term : terms) {
            int docsWithTerm = globalTermCounts.getOrDefault(term, 0);
            double idf = Math.log10((double) allFiles.size() / Math.max(1, docsWithTerm));
            globalIdfs.put(term, idf);
        }

        // --- Phase 2: Final Scoring ---
        List<SearchResponse.DocumentResult> finalResults = new ArrayList<>();
        currentFileIndex = 0;
        for (String address : workerList) {
            int count = Math.min(filesPerWorker, allFiles.size() - currentFileIndex);
            if (count <= 0) break;

            CalculationRequest request = CalculationRequest.newBuilder()
                    .addAllTerms(terms)
                    .putAllGlobalIdfs(globalIdfs)
                    .setStartIndex(currentFileIndex)
                    .setCount(count)
                    .build();
            try {
                SearchResponse response = stubs.get(address).getFinalScores(request);
                finalResults.addAll(response.getResultsList());
            } catch (Exception e) { System.err.println("Worker " + address + " Phase 2 error"); }
            currentFileIndex += count;
        }

        // --- Sort and Show Top 10 ---
        finalResults.sort((a, b) -> Double.compare(b.getScore(), a.getScore()));
        System.out.println("\n--- Search Results ---");
        for (int i = 0; i < Math.min(10, finalResults.size()); i++) {
            System.out.println((i + 1) + ". " + finalResults.get(i).getDocumentName() + " (Score: " + finalResults.get(i).getScore() + ")");
        }
    }
}