package com.distributed.search.logic;

import java.nio.charset.CharacterCodingException;
import java.nio.charset.StandardCharsets;
import java.nio.file.Files;
import java.nio.file.Path;
import java.io.IOException;
import java.util.*;
import java.util.stream.Collectors;

public class TFIDFCalculator {

    /**
     * تقرأ محتوى الملف وتحوّله إلى قائمة كلمات بشكل آمن
     * - بدون regex خاطئ
     * - يدعم Unicode (عربي / إنجليزي)
     * - لا يرمي Exceptions تكسر gRPC
     */
    public static List<String> getWordsFromDocument(Path filePath) {
        try {
            // قراءة الملف كنص UTF-8
            String content = Files.readString(filePath, StandardCharsets.ISO_8859_1);

            // أحرف صغيرة
            content = content.toLowerCase(Locale.ROOT);

            // استبدال أي شيء ليس حرفًا أو رقمًا بمسافة
            // \p{L} = أي حرف (أي لغة)
            // \p{N} = أي رقم
            String cleaned = content.replaceAll("[^\\p{L}\\p{N}]+", " ");

            // تقسيم على المسافات
            String[] tokens = cleaned.trim().split("\\s+");

            return Arrays.stream(tokens)
                    .filter(s -> !s.isEmpty())
                    .collect(Collectors.toList());

        } catch (CharacterCodingException e) {
            System.err.println("Character encoding error reading file: " + filePath);
            return Collections.emptyList();
        } catch (IOException e) {
            System.err.println("Error reading file: " + filePath + " - " + e.getMessage());
            return Collections.emptyList();
        } catch (Exception e) {
            System.err.println("Unexpected error processing file: " + filePath);
            e.printStackTrace();
            return Collections.emptyList();
        }
    }

    /**
     * Term Frequency
     */
    public static double calculateTermFrequency(List<String> words, String term) {
        if (words == null || words.isEmpty()) return 0.0;

        long count = words.stream()
                .filter(word -> word.equalsIgnoreCase(term))
                .count();

        return (double) count / words.size();
    }

    /**
     * Inverse Document Frequency
     */
    public static double calculateIdf(int totalDocuments, int documentsWithTerm) {
        if (documentsWithTerm <= 0 || totalDocuments <= 0) return 0.0;
        return Math.log10((double) totalDocuments / documentsWithTerm);
    }
}
