package com.distributed.search.web;

import com.distributed.search.grpc.SearchClient;
import com.distributed.search.logic.FileManager;
import com.distributed.search.model.SearchResponse;
import com.google.gson.Gson;
import com.sun.net.httpserver.HttpExchange;
import com.sun.net.httpserver.HttpServer;

import java.io.IOException;
import java.io.OutputStream;
import java.net.InetSocketAddress;
import java.util.Arrays;
import java.util.List;
import java.util.concurrent.Executors;

/**
 * Internal HTTP Server running on the Leader node.
 * It receives search queries from the Frontend and returns results in JSON format.
 */
public class CoordinatorWebServer {
    private static final String SEARCH_ENDPOINT = "/search";
    private final int port;
    private final SearchClient searchClient;
    private final String storageDir;
    private HttpServer server;
    private final Gson gson = new Gson();

    public CoordinatorWebServer(int port, SearchClient searchClient, String storageDir) {
        this.port = port;
        this.searchClient = searchClient;
        this.storageDir = storageDir;
    }

    public void startServer() {
        try {
            this.server = HttpServer.create(new InetSocketAddress(port), 0);
            server.createContext(SEARCH_ENDPOINT, this::handleSearchRequest);
            server.setExecutor(Executors.newFixedThreadPool(8));
            server.start();
            System.out.println("Leader Internal Web Server started on port: " + port);
        } catch (IOException e) {
            e.printStackTrace();
        }
    }

    public void stopServer() {
        if (server != null) server.stop(0);
    }

    /**
     * Handles search requests from the Frontend.
     * Expects query param: /search?query=word1+word2
     */
    private void handleSearchRequest(HttpExchange exchange) throws IOException {
        if (!exchange.getRequestMethod().equalsIgnoreCase("get")) {
            exchange.close();
            return;
        }

        // Extract query from URI
        String query = exchange.getRequestURI().getQuery().split("=")[1];
        List<String> terms = Arrays.asList(query.toLowerCase().split("\\+"));

        System.out.println("Internal API received query: " + terms);

        // Execute distributed search
        List<String> allFiles = FileManager.getSortedDocumentNames(storageDir);
        List<SearchResponse.DocumentResult> results = searchClient.performSearch(terms, allFiles);

        // Convert results to JSON using GSON
        String jsonResponse = gson.toJson(results);

        sendResponse(jsonResponse.getBytes(), exchange);
    }

    private void sendResponse(byte[] responseBytes, HttpExchange exchange) throws IOException {
        exchange.getResponseHeaders().add("Content-Type", "application/json");
        exchange.sendResponseHeaders(200, responseBytes.length);
        OutputStream outputStream = exchange.getResponseBody();
        outputStream.write(responseBytes);
        outputStream.flush();
        outputStream.close();
    }
}