package com.distributed.search.cluster;

import org.apache.zookeeper.*;
import org.apache.zookeeper.data.Stat;
import java.util.Collections;
import java.util.List;

/**
 * Implements the Fault-Tolerant Leader Election using Zookeeper.
 * Uses Ephemeral Sequential nodes to avoid 'Herd Effect'.
 */
public class LeaderElection implements Watcher {
    private static final String ELECTION_NAMESPACE = "/election";
    private final ZooKeeper zooKeeper;
    private final OnElectionCallback onElectionCallback;
    private String currentZnodeName;

    public LeaderElection(ZooKeeper zooKeeper, OnElectionCallback onElectionCallback) {
        this.zooKeeper = zooKeeper;
        this.onElectionCallback = onElectionCallback;
        ensureElectionNamespaceExists();
    }

    private void ensureElectionNamespaceExists() {
        try {
            if (zooKeeper.exists(ELECTION_NAMESPACE, false) == null) {
                zooKeeper.create(ELECTION_NAMESPACE, new byte[]{}, ZooDefs.Ids.OPEN_ACL_UNSAFE, CreateMode.PERSISTENT);
            }
        } catch (KeeperException | InterruptedException e) {
            e.printStackTrace();
        }
    }

    /**
     * Volunteers for leadership by creating an ephemeral sequential znode.
     */
    public void volunteerForLeadership() throws KeeperException, InterruptedException {
        String znodePrefix = ELECTION_NAMESPACE + "/c_";
        String znodeFullPath = zooKeeper.create(znodePrefix, new byte[]{},
                ZooDefs.Ids.OPEN_ACL_UNSAFE, CreateMode.EPHEMERAL_SEQUENTIAL);

        System.out.println("Znode created: " + znodeFullPath);
        this.currentZnodeName = znodeFullPath.replace(ELECTION_NAMESPACE + "/", "");
    }

    /**
     * Determines the leader based on the smallest sequence number.
     * If not the leader, watches the predecessor node for failures.
     */
    public void reelectLeader() throws KeeperException, InterruptedException {
        Stat predecessorStat = null;
        String predecessorName = "";

        // Loop to ensure we get a valid predecessor to watch
        while (predecessorStat == null) {
            List<String> children = zooKeeper.getChildren(ELECTION_NAMESPACE, false);
            Collections.sort(children);

            String smallestChild = children.get(0);

            if (smallestChild.equals(currentZnodeName)) {
                System.out.println("I am the Leader (Coordinator)");
                onElectionCallback.onElectedToBeLeader();
                return;
            } else {
                System.out.println("I am a Worker");
                int predecessorIndex = children.indexOf(currentZnodeName) - 1;
                predecessorName = children.get(predecessorIndex);
                // Watch the node that is exactly before us in the sequence
                predecessorStat = zooKeeper.exists(ELECTION_NAMESPACE + "/" + predecessorName, this);
            }
        }

        onElectionCallback.onWorker();
        System.out.println("Watching predecessor znode: " + predecessorName);
        System.out.println();
    }

    /**
     * Zookeeper Watcher event handler.
     * Re-triggers election if the watched predecessor node is deleted.
     */
    @Override
    public void process(WatchedEvent event) {
        if (event.getType() == Event.EventType.NodeDeleted) {
            try {
                reelectLeader();
            } catch (KeeperException | InterruptedException e) {
                e.printStackTrace();
            }
        }
    }
}