package com.distributed.search.grpc;

import com.distributed.search.model.*; // Generated Protobuf classes
import com.distributed.search.logic.*; // Custom logic classes (FileManager, TFIDFCalculator)
import io.grpc.stub.StreamObserver;
import java.io.IOException;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

/**
 * Implementation of the SearchService gRPC service.
 * This class handles the actual text processing logic on the Worker nodes.
 */
public class SearchServiceImpl extends SearchServiceGrpc.SearchServiceImplBase {

    private final String sharedDirectoryPath;

    public SearchServiceImpl(String sharedDirectoryPath) {
        this.sharedDirectoryPath = sharedDirectoryPath;
    }

    /**
     * Phase 1: Count document occurrences for each search term.
     * This information is sent back to the Coordinator to calculate Global IDF.
     */
    @Override
    public void getDocumentStats(StatRequest request, StreamObserver<StatResponse> responseObserver) {
        List<String> terms = request.getTermsList();
        int startIndex = request.getStartIndex();
        int count = request.getCount();

        // Retrieve sorted file names to ensure consistency across the cluster
        List<String> allFiles = FileManager.getSortedDocumentNames(sharedDirectoryPath);

        // Determine the sub-list of files assigned to this specific worker
        int endIndex = Math.min(startIndex + count, allFiles.size());
        List<String> assignedFiles = allFiles.subList(startIndex, endIndex);

        Map<String, Integer> termToDocCount = new HashMap<>();

        // Process each search term
        for (String term : terms) {
            int docsWithTerm = 0;
            for (String docName : assignedFiles) {
                try {
                    String content = FileManager.readDocumentContent(sharedDirectoryPath, docName);
                    List<String> words = TFIDFCalculator.getWordsFromDocument(content);

                    // Check if the document contains the term (case-insensitive)
                    if (words.contains(term.toLowerCase())) {
                        docsWithTerm++;
                    }
                } catch (IOException e) {
                    System.err.println("Error reading file: " + docName + " - " + e.getMessage());
                }
            }
            termToDocCount.put(term, docsWithTerm);
        }

        // Build the gRPC response
        StatResponse response = StatResponse.newBuilder()
                .putAllTermToDocumentCount(termToDocCount)
                .build();

        // Send the response and close the stream
        responseObserver.onNext(response);
        responseObserver.onCompleted();
    }

    /**
     * Phase 2: Calculate final TF-IDF scores for the assigned documents.
     * The scores are calculated using the Global IDF provided by the Coordinator.
     */
    @Override
    public void getFinalScores(CalculationRequest request, StreamObserver<SearchResponse> responseObserver) {
        List<String> terms = request.getTermsList();
        Map<String, Double> globalIdfs = request.getGlobalIdfsMap();
        int startIndex = request.getStartIndex();
        int count = request.getCount();

        // Ensure the worker processes the exact same set of files as in Phase 1
        List<String> allFiles = FileManager.getSortedDocumentNames(sharedDirectoryPath);
        int endIndex = Math.min(startIndex + count, allFiles.size());
        List<String> assignedFiles = allFiles.subList(startIndex, endIndex);

        SearchResponse.Builder responseBuilder = SearchResponse.newBuilder();

        // Calculate score for each assigned document
        for (String docName : assignedFiles) {
            double docScore = 0.0;
            try {
                String content = FileManager.readDocumentContent(sharedDirectoryPath, docName);
                List<String> words = TFIDFCalculator.getWordsFromDocument(content);

                for (String term : terms) {
                    // Calculate local Term Frequency (TF)
                    double tf = TFIDFCalculator.calculateTermFrequency(words, term);

                    // Retrieve the Global IDF sent by the Coordinator
                    double idf = globalIdfs.getOrDefault(term, 0.0);

                    // Accumulate the final score for the document
                    docScore += (tf * idf);
                }

                // Add document result to the response
                responseBuilder.addResults(SearchResponse.DocumentResult.newBuilder()
                        .setDocumentName(docName)
                        .setScore(docScore)
                        .build());

            } catch (IOException e) {
                System.err.println("Error calculating score for: " + docName + " - " + e.getMessage());
            }
        }

        // Send the final results back to the Coordinator
        responseObserver.onNext(responseBuilder.build());
        responseObserver.onCompleted();
    }
}