package com.distributed.search;

import com.distributed.search.cluster.LeaderElection;
import com.distributed.search.cluster.OnElectionAction;
import com.distributed.search.cluster.ServiceRegistry;
import com.distributed.search.grpc.SearchClient;
import com.distributed.search.logic.FileManager;
import org.apache.zookeeper.KeeperException;
import org.apache.zookeeper.WatchedEvent;
import org.apache.zookeeper.Watcher;
import org.apache.zookeeper.ZooKeeper;
import com.distributed.search.cluster.LeaderHttpRegistry;
import com.distributed.search.model.SearchResponse;
import java.io.IOException;
import java.util.Arrays;
import java.util.List;
import java.util.Scanner;

/**
 * Main Entry point for the Distributed Search Engine.
 */
public class Application implements Watcher {
    private static final String ZOOKEEPER_ADDRESS = "192.168.39.250:2181"; // Change if ZK is on another machine
    private static final int SESSION_TIMEOUT = 3000;
    private static final int DEFAULT_PORT = 8080;
    private static final String STORAGE_DIR = "storage"; // Folder containing .txt files

    private ZooKeeper zooKeeper;
    private static boolean isLeader = false;

    public static void main(String[] args) throws IOException, InterruptedException, KeeperException {
        // 1. Force the system to use IPv4 and disable native transports that cause 'unix' errors
        System.setProperty("io.grpc.netty.shaded.io.netty.transport.noNative", "true");
        System.setProperty("java.net.preferIPv4Stack", "true");

        // Use port from arguments to allow multiple instances on the same machine
        int currentServerPort = args.length == 1 ? Integer.parseInt(args[0]) : DEFAULT_PORT;

        Application application = new Application();
        ZooKeeper zooKeeper = application.connectToZookeeper();

        ServiceRegistry serviceRegistry = new ServiceRegistry(zooKeeper);
        LeaderHttpRegistry leaderHttpRegistry = new LeaderHttpRegistry(zooKeeper); // Discovery Bridge
        SearchClient searchClient = new SearchClient();

        // Action to take when election completes
        OnElectionAction onElectionAction = new OnElectionAction(
                serviceRegistry,
                leaderHttpRegistry,
                searchClient,
                currentServerPort) {
            @Override
            public void onElectedToBeLeader() {
                super.onElectedToBeLeader();
                isLeader = true;
            }
        };

        LeaderElection leaderElection = new LeaderElection(zooKeeper, onElectionAction);
        leaderElection.volunteerForLeadership();
        leaderElection.reelectLeader();

        // If this node is the Leader, enter the Search Loop
        application.enterSearchLoop(serviceRegistry, searchClient);

        application.close();
    }

    /**
     * The Main UI Loop: Only active for the Leader/Coordinator node.
     */
    private void enterSearchLoop(ServiceRegistry serviceRegistry, SearchClient searchClient) throws InterruptedException {
        Scanner scanner = new Scanner(System.in);

        while (true) {
            if (isLeader) {
                // If Leader: Keep providing the Console UI for local testing
                System.out.println("\n[Leader Mode] Enter query (Internal API is also listening...):");
                if (!scanner.hasNextLine()) break;
                String input = scanner.nextLine();

                if (input.equalsIgnoreCase("exit")) break;
                if (input.trim().isEmpty()) continue;

                List<String> workers = serviceRegistry.getAllServiceAddresses();
                if (workers.isEmpty()) {
                    System.out.println("Wait for workers to join...");
                    continue;
                }

                List<String> terms = Arrays.asList(input.toLowerCase().split("\\s+"));
                List<String> allFiles = FileManager.getSortedDocumentNames(STORAGE_DIR);

                searchClient.updateWorkers(workers);
                // Perform search and get results back
                List<SearchResponse.DocumentResult> results = searchClient.performSearch(terms, allFiles);

                // Print top results in console
                results.stream().limit(10).forEach(r ->
                        System.out.println(r.getDocumentName() + " (Score: " + r.getScore() + ")"));

            } else {
                // If Worker: Just sleep and keep connection alive (HTTP and gRPC servers run in background threads)
                Thread.sleep(10000);
            }
        }
    }

    public ZooKeeper connectToZookeeper() throws IOException {
        this.zooKeeper = new ZooKeeper(ZOOKEEPER_ADDRESS, SESSION_TIMEOUT, this);
        return zooKeeper;
    }

    private void close() throws InterruptedException {
        this.zooKeeper.close();
    }

    @Override
    public void process(WatchedEvent event) {
        if (event.getType() == Event.EventType.None && event.getState() == Event.KeeperState.SyncConnected) {
            System.out.println("Successfully connected to Zookeeper");
        }
    }
}