package worker;

import org.apache.zookeeper.*;
import common.ZkConfig;
import java.io.IOException;
import java.util.Random;
import java.util.concurrent.locks.LockSupport;

public class TransientWorker {

    private static final float CHANCE_TO_FAIL = 0.1F;
    private final Random random = new Random();
    private ZooKeeper zooKeeper;
    private final String nodeId;

    public TransientWorker(String nodeId) {
        this.nodeId = nodeId;
    }

    public static void main(String[] args) {
        if (args.length < 1) {
            System.out.println("Usage: java -jar TransientWorker.jar <nodeId>");
            System.exit(1);
        }

        String nodeId = args[0];
        TransientWorker worker = new TransientWorker(nodeId);

        try {
            worker.connectToZookeeper();
            worker.work();
        } catch (Exception e) {
            System.err.println("Worker on node [" + nodeId + "] crashed: " + e.getMessage());
            System.exit(1);
        }
    }

    public void connectToZookeeper() throws IOException {
        // We can pass null for the watcher since the worker doesn't need to respond to events
        this.zooKeeper = new ZooKeeper(ZkConfig.ZOOKEEPER_ADDRESS, ZkConfig.SESSION_TIMEOUT, null);
    }

    public void work() throws KeeperException, InterruptedException {
        registerWorkerInCluster();

        while (true) {
            System.out.println("Worker on node [" + nodeId + "] is performing calculations...");
            LockSupport.parkNanos(2_000_000_000L); // 2 seconds

            if (random.nextFloat() < CHANCE_TO_FAIL) {
                System.out.println("Critical error simulated.");
                throw new RuntimeException("Oops! Worker crashed.");
            }
        }
    }

    private void registerWorkerInCluster() throws KeeperException, InterruptedException {
        String path = ZkConfig.WORKERS_PARENT_PATH + "/worker_";
        byte[] data = nodeId.getBytes();

        // Create ephemeral sequential node to signal presence
        zooKeeper.create(path, data, ZooDefs.Ids.OPEN_ACL_UNSAFE, CreateMode.EPHEMERAL_SEQUENTIAL);
    }
}