package common;

import java.io.BufferedWriter;
import java.io.FileWriter;
import java.io.IOException;
import java.time.LocalDateTime;
import java.time.format.DateTimeFormatter;
import java.util.concurrent.BlockingQueue;
import java.util.concurrent.LinkedBlockingQueue;

public class LoggerUtil {
    private static final DateTimeFormatter formatter = DateTimeFormatter.ofPattern("yyyy-MM-dd HH:mm:ss");

    // Memory queue to hold log messages
    private static final BlockingQueue<String> logQueue = new LinkedBlockingQueue<>();
    private static BufferedWriter writer;

    static {
        try {
            // Keep the file open for better performance
            writer = new BufferedWriter(new FileWriter(ZkConfig.CLUSTER_LOG_FILE, true));

            // Start a background thread to process logs asynchronously
            Thread logThread = new Thread(() -> {
                try {
                    while (true) {
                        String message = logQueue.take(); // Wait for a message in the queue
                        writer.write(message);
                        writer.newLine();
                        writer.flush(); // Flush to disk without closing
                    }
                } catch (InterruptedException | IOException e) {
                    Thread.currentThread().interrupt();
                }
            });

            logThread.setDaemon(true); // Let it close when the main application stops
            logThread.start();

        } catch (IOException e) {
            System.err.println("Failed to initialize fast logger: " + e.getMessage());
        }
    }

    /**
     * Rapidly logs an event by adding it to an internal queue.
     * This method is non-blocking and returns almost instantly.
     */
    public static void log(String message) {
        String timestampedMessage = String.format("[%s] %s", LocalDateTime.now().format(formatter), message);

        // Print to console (fast)
        System.out.println(timestampedMessage);

        // Offer the message to the background logging thread
        logQueue.offer(timestampedMessage);
    }
}