package org.example;

import java.util.concurrent.BlockingQueue;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.atomic.AtomicInteger;

public class WorkerConsumer implements Runnable {
    private final BlockingQueue<String> inputQueue;
    private final BlockingQueue<String> outputQueue;
    private final int workerId;
    private final AtomicInteger processedCount = new AtomicInteger(0);

    public WorkerConsumer(BlockingQueue<String> inputQueue,
                          BlockingQueue<String> outputQueue,
                          int workerId) {
        this.inputQueue = inputQueue;
        this.outputQueue = outputQueue;
        this.workerId = workerId;
    }

    @Override
    public void run() {

        Main.getActiveWorkers().incrementAndGet();
        System.out.println(" the worker " + workerId + ": started");

        try {
            while (true) {

                String line = inputQueue.poll(1, TimeUnit.SECONDS);

                if (line != null) {

                    processLine(line);
                } else if (Main.isProducerFinished() && inputQueue.isEmpty()) {
                    break;
                }
            }
        } catch (InterruptedException e) {
            System.err.println("️ the worker " + workerId + ": has interrupted");
            Thread.currentThread().interrupt();
        } finally {
            Main.getActiveWorkers().decrementAndGet();
            System.out.println(" the worker " + workerId + ": finished after the processing " +
                    processedCount.get() + "line");
        }
    }

    private void processLine(String line) throws InterruptedException {
        System.out.println(" the worker " + workerId + "  is processing: " + shortenText(line, 30));


        Thread.sleep(100 + (int)(Math.random() * 100));


        String processedLine = applyStemming(line);


        outputQueue.put(processedLine);

        int count = processedCount.incrementAndGet();
        System.out.println("the worker " + workerId + ": finished the line " + count + " - " +
                shortenText(processedLine, 30));
    }

    private String applyStemming(String line) {

        String[] words = line.split("\\s+");
        StringBuilder result = new StringBuilder();

        for (String word : words) {
            if (!word.isEmpty()) {
                String stemmedWord = stemWord(word.toLowerCase());
                result.append(stemmedWord).append(" ");
            }
        }

        return result.toString().trim();
    }

    private String stemWord(String word) {
        if (word.endsWith("ing")) {
            return word.substring(0, word.length() - 3);
        } else if (word.endsWith("ed")) {
            return word.substring(0, word.length() - 2);
        } else if (word.endsWith("s")) {
            return word.substring(0, word.length() - 1);
        } else if (word.endsWith("ly")) {
            return word.substring(0, word.length() - 2);
        }
        return word;
    }

    private String shortenText(String text, int maxLength) {
        if (text.length() <= maxLength) {
            return text;
        }
        return text.substring(0, maxLength) + "...";
    }
}