package org.example;

import java.io.*;
import java.nio.file.*;
import java.util.concurrent.*;
import java.util.concurrent.atomic.AtomicInteger;

public class Main {
    // Variables shared by all threads
    private static volatile boolean producerFinished = false;
    private static final AtomicInteger activeWorkers = new AtomicInteger(0);

    public static void main(String[] args) {
        System.out.println("Starting word processing system...");


        createSampleInputFile();

        // Set up queues - using LinkedBlockingQueue as in the appendix
        BlockingQueue<String> rawLinesQueue = new LinkedBlockingQueue<>(100);
        BlockingQueue<String> processedLinesQueue = new LinkedBlockingQueue<>(100);


        ExecutorService executor = Executors.newFixedThreadPool(6);


        long startTime = System.currentTimeMillis();


        executor.submit(new Producer(rawLinesQueue, "input.txt"));


        for (int i = 1; i <= 3; i++) {
            executor.submit(new WorkerConsumer(rawLinesQueue, processedLinesQueue, i));
        }


        executor.submit(new WriterConsumer(processedLinesQueue, "output.txt"));


        executor.shutdown();

        try {

            if (executor.awaitTermination(2, TimeUnit.MINUTES)) {
                long endTime = System.currentTimeMillis();
                System.out.println("Processing completed successfully " + (endTime - startTime) + " ms");
            } else {
                System.err.println("️ Time runs out before all tasks are completed");
                executor.shutdownNow();
            }
        } catch (InterruptedException e) {
            System.err.println("The wait is interrupted");
            executor.shutdownNow();
            Thread.currentThread().interrupt();
        }


        displayResults();
    }

    public static boolean isProducerFinished() {
        return producerFinished;
    }

    public static void setProducerFinished(boolean finished) {
        producerFinished = finished;
    }

    public static AtomicInteger getActiveWorkers() {
        return activeWorkers;
    }

    private static void createSampleInputFile() {
        try (BufferedWriter writer = Files.newBufferedWriter(Paths.get("input.txt"))) {
            String[] sampleLines = {
                    "The quick brown foxes are jumping over the lazy dogs",
                    "Running in the beautiful gardens and playing with balls",
                    "Programming computers and learning new technologies",
                    "Artificial intelligence and machine learning algorithms",
                    "Natural language processing is an amazing field of study",
                    "Students are studying hard for their final examinations",
                    "The workers are working on important projects and developments",
                    "Beautiful flowers are growing in the wonderful garden",
                    "Scientists are researching new discoveries and innovations",
                    "Engineers are building amazing structures and constructions"
            };

            for (String line : sampleLines) {
                writer.write(line);
                writer.newLine();
            }
            System.out.println("The input.txt file was created successfully");
        } catch (IOException e) {
            System.err.println("Error creating file: " + e.getMessage());
        }
    }

    private static void displayResults() {
        try {
            System.out.println("\n" + "=".repeat(50));
            System.out.println("final Resault");
            System.out.println("=".repeat(50));

            System.out.println("\n Original file (input.txt):");
            System.out.println("-".repeat(30));
            Files.lines(Paths.get("input.txt")).forEach(System.out::println);

            System.out.println("\n Processed file (output.txt):");
            System.out.println("-".repeat(30));
            Files.lines(Paths.get("output.txt")).forEach(System.out::println);

        } catch (IOException e) {
            System.err.println("Error displaying results: " + e.getMessage());
        }
    }
}