package org.example;

import org.apache.zookeeper.KeeperException;
import org.apache.zookeeper.WatchedEvent;
import org.apache.zookeeper.Watcher;
import org.apache.zookeeper.ZooKeeper;
import org.apache.zookeeper.data.Stat;

import java.io.IOException;
import java.util.List;

public class WatchersAndTriggers implements Watcher {

    private static final String address = "192.168.184.101:2181";

    private static final int SESSION_TIMEOUT = 3000; // Dead client timeout
    private static final String TargetZnode = "/target_znode";

    private ZooKeeper zooKeeper;

    public static void main(String[] args) throws IOException, InterruptedException, KeeperException {
        WatchersAndTriggers watchersAndTriggers = new WatchersAndTriggers();

        watchersAndTriggers.connectToZookeeper();
        watchersAndTriggers.initialSetupAndWatchers();

        watchersAndTriggers.run();
        watchersAndTriggers.close();
        System.out.println("Closed Successfully");
    }

    private void close() throws InterruptedException {
        this.zooKeeper.close();
    }

    private void run() throws InterruptedException {
        synchronized (zooKeeper) {
            this.zooKeeper.wait();
        }
    }

    private void connectToZookeeper() throws IOException {
        this.zooKeeper = new ZooKeeper(address, SESSION_TIMEOUT, this);
    }

    /**
     * Function to set initial watchers (Initial Setup).
     */
    public void initialSetupAndWatchers() throws InterruptedException, KeeperException {
        // 1. Exist Watch: using "exists"
        Stat stat = zooKeeper.exists(TargetZnode, this);

        if (stat == null) {
            System.out.println(TargetZnode + " not exist. Waiting for creation.");
            return;
        }

        // 2. Data Watch: using "getData"
        byte[] data = zooKeeper.getData(TargetZnode, this, stat);

        // 3. Children Watch: using "getChildren"
        List<String> children = zooKeeper.getChildren(TargetZnode, this);

        System.out.println("Data: " + new String(data) + ", Children: " + children);
    }

    /**
     * The modified event processing function (PROCESS).
     * Selectively re-arms only the triggered watcher to avoid the race condition.
     */
    @Override
    public void process(WatchedEvent watchedEvent) {
        try {
            switch (watchedEvent.getType()) {
                case None:
                    if (watchedEvent.getState() == Event.KeeperState.SyncConnected) {
                        System.out.println("Successfully connected to Zookeeper");
                    } else if (watchedEvent.getState() == Event.KeeperState.Disconnected) {
                        synchronized (zooKeeper) {
                            System.out.println("Disconnected from Zookeeper");
                            zooKeeper.notifyAll();
                        }
                    } else if (watchedEvent.getState() == Event.KeeperState.Closed) {
                        System.out.println("Closed Successfully");
                    }
                    break;

                case NodeCreated:
                    System.out.println(TargetZnode + " Created");
                    // On creation: re-arm all watchers (Data, Children, and Exist)
                    // and fetch current data/children
                    rearmAllWatchersAfterCreation();
                    break;

                case NodeDeleted:
                    System.out.println(TargetZnode + " Deleted");
                    // On deletion: re-arm only the Exist Watch (in case the znode is recreated)
                    zooKeeper.exists(TargetZnode, this);
                    break;

                case NodeDataChanged:
                    System.out.println(TargetZnode + " DataChanged");
                    // Re-arm only the Data Watch
                    zooKeeper.getData(TargetZnode, this, new Stat());
                    break;

                case NodeChildrenChanged:
                    System.out.println(TargetZnode + " ChildrenChanged");
                    // Re-arm only the Children Watch
                    zooKeeper.getChildren(TargetZnode, this);
                    break;
            }
        } catch (InterruptedException e) {
            Thread.currentThread().interrupt();
        } catch (KeeperException e) {
            // Handle exceptions
            System.err.println("KeeperException during re-arming watchers: " + e.getMessage());
        }
    }

    /**
     * Helper function to get data and re-arm all watchers after node creation,
     * ensuring the latest state is captured.
     */
    private void rearmAllWatchersAfterCreation() throws InterruptedException, KeeperException {
        Stat stat = zooKeeper.exists(TargetZnode, this); // Re-arm Exist Watch
        if (stat != null) {
            byte[] data = zooKeeper.getData(TargetZnode, this, stat); // Re-arm Data Watch
            List<String> children = zooKeeper.getChildren(TargetZnode, this); // Re-arm Children Watch
            System.out.println("New Data: " + new String(data) + ", New Children: " + children);
        }
    }
}