package org.example;

import org.apache.zookeeper.*;
import org.apache.zookeeper.data.Stat;

import java.io.IOException;
import java.util.Collections;
import java.util.List;

public class ImprovedLeaderElection implements Watcher {
    // Make sure to update this address to match your ZooKeeper server address
    private static final String address = "192.168.184.103:2181";
    private static final int SESSION_TIMEOUT = 3000;
    private static final String ELECTION_NAMESPACE = "/election";
    private static final String ZNODE_PREFIX = ELECTION_NAMESPACE + "/c_";

    private String currentZnodeName;
    private ZooKeeper zooKeeper;

    public static void main(String[] args) throws IOException, InterruptedException, KeeperException {
        ImprovedLeaderElection leaderElection = new ImprovedLeaderElection();
        leaderElection.connectToZookeeper();

        // 1. Volunteer to create the candidacy node
        leaderElection.volunteerForLeadership();

        // 2. Apply the Sequential Watching algorithm
        leaderElection.electLeader();

        leaderElection.run();
        leaderElection.close();
        System.out.println("Closed successfully");
    }

    public void volunteerForLeadership() throws InterruptedException, KeeperException {
        // Create an Ephemeral Sequential znode
        String znodeFullPath = zooKeeper.create(ZNODE_PREFIX, new byte[]{}, ZooDefs.Ids.OPEN_ACL_UNSAFE, CreateMode.EPHEMERAL_SEQUENTIAL);

        System.out.println("Candidacy Znode created: " + znodeFullPath);
        this.currentZnodeName = znodeFullPath.replace(ELECTION_NAMESPACE + "/", "");
    }

    /**
     * The core function of the Sequential Watching algorithm.
     * Each node places a Watcher only on the node immediately preceding it.
     */
    public void electLeader() throws InterruptedException, KeeperException {
        // 1. Get the list of children without setting a watcher (false)
        List<String> children = zooKeeper.getChildren(ELECTION_NAMESPACE, false);
        Collections.sort(children);

        String smallestChild = children.get(0); // Potential Leader
        int myIndex = children.indexOf(currentZnodeName);

        if (smallestChild.equals(currentZnodeName)) {
            // I am the Leader: I monitor nothing
            System.out.println("\n-----------------------------------------");
            System.out.println("I am the LEADER: " + currentZnodeName);
            System.out.println("-----------------------------------------\n");
        } else {
            // I am a Follower: I must monitor only the node preceding me
            String smallerSiblingName = children.get(myIndex - 1);
            String smallerSiblingPath = ELECTION_NAMESPACE + "/" + smallerSiblingName;

            // **We only set a watcher on the previous node via exists**
            // The watcher is 'this', and will receive a NodeDeleted notification when the previous node falls (Herd Effect broken)
            Stat stat = zooKeeper.exists(smallerSiblingPath, this);

            if (stat == null) {
                // The previous node fell before the watcher was set, re-elect
                System.out.println("The previous node does not exist. Re-election...");
                electLeader();
            } else {
                System.out.println("I am a Follower. I monitor the previous node: " + smallerSiblingName);
            }
        }
    }

    private void close() throws InterruptedException {
        this.zooKeeper.close();
    }

    public void connectToZookeeper() throws IOException {
        this.zooKeeper = new ZooKeeper(address, SESSION_TIMEOUT, this);
    }

    public void run() throws InterruptedException {
        synchronized (zooKeeper) {
            zooKeeper.wait();
        }
    }

    /**
     * Event processing: Re-election is triggered only upon receiving a NodeDeleted notification.
     */
    @Override
    public void process(WatchedEvent watchedEvent) {
        switch (watchedEvent.getType()) {
            case None:
                if (watchedEvent.getState() == Event.KeeperState.SyncConnected) {
                    System.out.println("Connected to ZooKeeper successfully");
                } else if (watchedEvent.getState() == Event.KeeperState.Disconnected) {
                    synchronized (zooKeeper) {
                        System.out.println("The connection to ZooKeeper has been lost");
                        zooKeeper.notifyAll();
                    }
                } else if (watchedEvent.getState() == Event.KeeperState.Closed) {
                    System.out.println("Closed successfully");
                }
                break;

            // The only event that should trigger the re-election process
            case NodeDeleted:
                try {
                    // Only the node monitoring the deleted node will reach here,
                    // which breaks the synchronization (Herd Effect)
                    System.out.println("--- I was notified: The previous node has been deleted. Re-election... ---");
                    electLeader();
                } catch (InterruptedException e) {
                    throw new RuntimeException(e);
                } catch (KeeperException e) {
                    throw new RuntimeException(e);
                }
                break;

            default:
                // Ignore any other events that do not serve the election algorithm
                break;
        }
    }
}