package main

import (
	"context"
	"fmt"
	"io"
	"log"
	"net"
	"os"
	"time"

	//Import the code we generated
	"example.com/multiplication_service/calculatorpb"
	"google.golang.org/grpc"
)


const logFile = "multiplication_log.txt"


type server struct {
	calculatorpb.UnimplementedMultiplicationServiceServer
}

// 2. Apply the multiply function
func (*server) Multiply(ctx context.Context, req *calculatorpb.OperationRequest) (*calculatorpb.OperationResponse, error) {
	num1 := req.GetNum1()
	num2 := req.GetNum2()
	fmt.Printf("Received Multiply request: num1=%d, num2=%d\n", num1, num2)


	result := num1 * num2
	logMessage := fmt.Sprintf("Multiply Operation: %d * %d = %d\n", num1, num2, result)

	//---Recording the operation in the log---
	f, err := os.OpenFile(logFile, os.O_APPEND|os.O_CREATE|os.O_WRONLY, 0644)
	if err != nil {
		log.Printf("Failed to open log file: %v", err)
	} else {
		if _, err := f.WriteString(logMessage); err != nil {
			log.Printf("Failed to write to log file: %v", err)
		}
		f.Close()
	}

	// Returns the response
	return &calculatorpb.OperationResponse{Result: result}, nil
}

//3. Implement the log broadcast function (GetMultiplicationLog)
func (*server) GetMultiplicationLog(req *calculatorpb.GetLogRequest, stream calculatorpb.MultiplicationService_GetMultiplicationLogServer) error {
	fmt.Println("Received GetLog request. Streaming log...")

	f, err := os.Open(logFile)
	if err != nil {
		log.Printf("Log file not found: %v", err)
		return fmt.Errorf("log file not found")
	}
	defer f.Close()

	//Use buffer for reading
	buf := make([]byte, 1024)
	for {
		n, err := f.Read(buf)
		if err == io.EOF {
			break
		}
		if err != nil {
			log.Printf("Error reading log file: %v", err)
			return err
		}

		//Sending data as a “stream” to the client
		if err := stream.Send(&calculatorpb.LogEntry{Message: string(buf[:n])}); err != nil {
			log.Printf("Error sending stream: %v", err)
			return err
		}
		time.Sleep(100 * time.Millisecond) //Simple delay to simulate broadcast
	}
	return nil
}

func main() {
	// Determine the port
	port := "50052"
	lis, err := net.Listen("tcp", "0.0.0.0:"+port)
	if err != nil {
		log.Fatalf("Failed to listen: %v", err)
	}

	// Create a new gRPC server
	s := grpc.NewServer()

	// Register our service on the server
	calculatorpb.RegisterMultiplicationServiceServer(s, &server{})

	log.Printf("Multiplication Service server started on port %s...", port)

	// Run the server
	if err := s.Serve(lis); err != nil {
		log.Fatalf("Failed to serve: %v", err)
	}
}