
package com.example.grpc.calculator;


import io.grpc.ManagedChannel;
import io.grpc.ManagedChannelBuilder;
import io.grpc.StatusRuntimeException;
import java.util.Iterator;
import java.util.Scanner;

public class ClientGateway {

    // Create communication "channels" with services
    // Channel for collection service (Python) on port 50051
    private final ManagedChannel additionChannel;
    private final AdditionServiceGrpc.AdditionServiceBlockingStub additionStub;

    // Channel for the Go service on port 50052
    private final ManagedChannel multiplicationChannel;
    private final MultiplicationServiceGrpc.MultiplicationServiceBlockingStub multiplicationStub;

    public ClientGateway(String addr1, int port1, String addr2, int port2) {
        //Set up the sum channel
        this.additionChannel = ManagedChannelBuilder.forAddress(addr1, port1)
                .usePlaintext() // Use an unencrypted connection (for testing)
                .build();
        this.additionStub = AdditionServiceGrpc.newBlockingStub(additionChannel);

        // Set up the multiplication channel
        this.multiplicationChannel = ManagedChannelBuilder.forAddress(addr2, port2)
                .usePlaintext()
                .build();
        this.multiplicationStub = MultiplicationServiceGrpc.newBlockingStub(multiplicationChannel);
    }

    // Function to close channels
    public void shutdown() {
        additionChannel.shutdown();
        multiplicationChannel.shutdown();
    }

    // --- 1. Function calling addition and multiplication operation ---
    public void performOperations(int num1, int num2) {
        System.out.println("Calling services with num1=" + num1 + ", num2=" + num2);

        // Create the request message
        OperationRequest request = OperationRequest.newBuilder()
                .setNum1(num1)
                .setNum2(num2)
                .build();

        // call collection service (Python)
        try {
            OperationResponse addResponse = additionStub.add(request);
            System.out.println("Addition Result (from Python): " + addResponse.getResult());
        } catch (StatusRuntimeException e) {
            System.err.println("RPC failed (Addition): " + e.getStatus());
        }

        // call multiplication service (Go)
        try {
            OperationResponse multiplyResponse = multiplicationStub.multiply(request);
            System.out.println("Multiplication Result (from Go): " + multiplyResponse.getResult());
        } catch (StatusRuntimeException e) {
            System.err.println("RPC failed (Multiplication): " + e.getStatus());
        }
    }

    // --- 2. Collection record request function (Streaming) ---
    public void getAdditionLog() {
        System.out.println("\n--- Fetching Addition Log (Python) ---");
        GetLogRequest request = GetLogRequest.newBuilder().build();
        try {
            Iterator<LogEntry> logs = additionStub.getAdditionLog(request);
            while (logs.hasNext()) {
                System.out.println(logs.next().getMessage());
            }
        } catch (StatusRuntimeException e) {
            System.err.println("RPC failed (Get Addition Log): " + e.getStatus());
        }
    }

    // --- 3. Record multiplication request function (Streaming) ---
    public void getMultiplicationLog() {
        System.out.println("\n--- Fetching Multiplication Log (Go) ---");
        GetLogRequest request = GetLogRequest.newBuilder().build();
        try {
            Iterator<LogEntry> logs = multiplicationStub.getMultiplicationLog(request);
            while (logs.hasNext()) {
                System.out.println(logs.next().getMessage());
            }
        } catch (StatusRuntimeException e) {
            System.err.println("RPC failed (Get Multiplication Log): " + e.getStatus());
        }
    }



    //---Main function of client operation---
    public static void main(String[] args) {
        // Create the client and specify the service addresses
        ClientGateway client = new ClientGateway("localhost", 50051, "localhost", 50052);

        // Create a Scanner to read from the user
        Scanner scanner = new Scanner(System.in);

        try {
            while (true) {
                System.out.println("\n---------------------------------");
                System.out.println("Select an option:");
                System.out.println("1: Perform Operations (Add & Multiply)");
                System.out.println("2: Get Logs (Streaming)");
                System.out.println("0: Exit");
                System.out.print("Enter your choice: ");

                int choice = scanner.nextInt();

                if (choice == 1) {
                    // --- 1. Receive numbers from user ---
                    System.out.print("Enter number 1: ");
                    int num1 = scanner.nextInt();
                    System.out.print("Enter number 2: ");
                    int num2 = scanner.nextInt();

                    // --- 2. Sending numbers to services ---
                    client.performOperations(num1, num2);

                } else if (choice == 2) {
                    // --- 3. Request records (Streaming) ---
                    client.getAdditionLog();
                    client.getMultiplicationLog();

                } else if (choice == 0) {
                    System.out.println("Exiting...");
                    break;
                } else {
                    System.out.println("Invalid choice. Please try again.");
                }
            }
        } finally {
            //Close the connections and Scanner when finished
            client.shutdown();
            scanner.close();
        }
    }
}