package com.search.project;

import com.search.project.proto.SearchRequest;
import com.search.project.proto.SearchResponse;
import com.search.project.proto.SearchServiceGrpc;
import io.grpc.ManagedChannel;
import io.grpc.ManagedChannelBuilder;
import org.apache.curator.framework.CuratorFramework;
import java.io.File;
import java.net.InetAddress;
import java.util.ArrayList;
import java.util.List;

public class SearchCoordinator {
    private static final String DATA_DIR = "src/main/resources";

    public static List<SearchResponse.DocumentScore> performDistributedSearch(String query, CuratorFramework client, ServiceRegistry registry) {
        List<SearchResponse.DocumentScore> finalResults = new ArrayList<>();
        if (query == null || query.isEmpty()) return finalResults;

        try {
            // 1. جلب كل العمال المسجلين في زوكيبر
            List<String> allWorkerNodes = registry.getAllWorkers();

            // تحديد العنوان الكامل للقائد الحالي (IP + Port) للاستبعاد
            String myIp = InetAddress.getLocalHost().getHostAddress();
            String myFullAddress = myIp + ":" + WorkerNode.serverPort;

            // قائمة لتخزين عناوين العمال الفعليين الذين سنرسل لهم الطلبات
            List<String> validWorkersAddresses = new ArrayList<>();

            for (String nodeName : allWorkerNodes) {
                byte[] data = client.getData().forPath("/service_registry/" + nodeName);
                String address = new String(data);

                //  الاستبعاد يعتمد على العنوان الكامل وليس الـ IP فقط
                if (address.equals(myFullAddress)) {
                    System.out.println("[Coordinator]  أنا القائد: " + address);
                    continue;
                }
                validWorkersAddresses.add(address);
            }

            // 2. تجهيز الملفات
            File dir = new File(DATA_DIR);
            File[] files = dir.listFiles((d, name) -> name.endsWith(".txt"));

            if (files == null || validWorkersAddresses.isEmpty()) {
                System.out.println("[Coordinator] لا يوجد ملفات أو لا يوجد عمال متاحون حالياً!");
                return finalResults;
            }

            List<String> allFileNames = new ArrayList<>();
            for (File f : files) allFileNames.add(f.getName());

            int totalFiles = allFileNames.size();
            int numWorkers = validWorkersAddresses.size();
            int startIndex = 0;

            // 3. توزيع الملفات على العمال المتاحين
            List<String> filesRemaining = new ArrayList<>(allFileNames); // قائمة الملفات
            int workersLeft = validWorkersAddresses.size(); // عدد العمال

            for (String workerAddress : validWorkersAddresses) {
                if (filesRemaining.isEmpty()) break;

                // حساب حصة العامل الحالي من الملفات "المتبقية" فقط
                int chunkSize = (int) Math.ceil((double) filesRemaining.size() / workersLeft);
                int toIndex = Math.min(chunkSize, filesRemaining.size());
                List<String> workerFiles = new ArrayList<>(filesRemaining.subList(0, toIndex));

                String[] parts = workerAddress.split(":");
                String host = parts[0];
                int port = Integer.parseInt(parts[1]);

                System.out.println("[Coordinator] محاولة إرسال " + workerFiles.size() + " ملف إلى العامل: " + workerAddress);

                ManagedChannel channel = ManagedChannelBuilder.forAddress(host, port)
                        .usePlaintext()
                        .build();

                try {
                    SearchServiceGrpc.SearchServiceBlockingStub stub = SearchServiceGrpc.newBlockingStub(channel);

                    SearchResponse response = stub.processSearch(
                            SearchRequest.newBuilder()
                                    .setQuery(query)
                                    .addAllFileNames(workerFiles)
                                    .build()
                    );


                    finalResults.addAll(response.getResultsList());
                    // إزالة الملفات التي تمت معالجتها بنجاح من القائمة المتبقية
                    filesRemaining.removeAll(workerFiles);
                    System.out.println("[Coordinator] نجح العامل: " + workerAddress);

                } catch (Exception e) {

                    System.err.println("[Coordinator] فشل العامل " + workerAddress + " وسيتم تحويل ملفاته لعامل آخر.");
                } finally {
                    channel.shutdown();
                    workersLeft--; // تنقيص عدد العمال المتبقيين للمحاولة القادمة
                }
            }

            // 4. حساب الـ IDF والترتيب النهائي
            // IDF = log(عدد الملفات الكلي / عدد الملفات التي تحتوي الكلمة)
            double idf = Math.log((double) totalFiles / (finalResults.isEmpty() ? 1 : finalResults.size())) ;

            List<SearchResponse.DocumentScore> sortedWithIdf = new ArrayList<>();
            for (SearchResponse.DocumentScore ds : finalResults) {
                sortedWithIdf.add(SearchResponse.DocumentScore.newBuilder()
                        .setDocName(ds.getDocName())
                        .setScore(ds.getScore() * idf).build());
            }

            // ترتيب تنازلي حسب السكور النهائي (TF * IDF)
            sortedWithIdf.sort((a, b) -> Double.compare(b.getScore(), a.getScore()));

            return sortedWithIdf;

        } catch (Exception e) {
            e.printStackTrace();
        }
        return finalResults;
    }
}