package com.search.project;

import com.search.project.proto.SearchRequest;
import com.search.project.proto.SearchResponse;
import com.search.project.proto.SearchServiceGrpc;
import io.grpc.*;
import org.apache.curator.framework.CuratorFramework;
import org.apache.curator.framework.CuratorFrameworkFactory;
import org.apache.curator.retry.ExponentialBackoffRetry;
import java.net.InetAddress;
import java.net.ServerSocket;
import java.util.List;
import java.util.Scanner;

public class Application {
    private static final String ZK_ADDRESS = "172.29.3.101:2181";

    public static void main(String[] args) throws Exception {
        CuratorFramework client = CuratorFrameworkFactory.newClient(ZK_ADDRESS, new ExponentialBackoffRetry(1000, 3));
        client.start();

        ServiceRegistry registry = new ServiceRegistry(client);

        // بدء انتخاب القائد
        registry.startLeaderElection(() -> {
            System.out.println("[Coordinator] أنا الآن القائد والمنسق للنظام.");
            new Thread(() -> startSearchInterface(client, registry)).start();
        });

        // تشغيل كعامل (Worker) في الخلفية دائماً
        startWorkerNode(registry);
    }

    private static void startSearchInterface(CuratorFramework client, ServiceRegistry registry) {
        Scanner scanner = new Scanner(System.in);
        while (true) {
            System.out.print("\n>>> أدخل كلمة البحث (أو 'exit'): ");
            String query = scanner.nextLine();
            if (query.equalsIgnoreCase("exit")) break;

            try {
                List<String> workers = registry.getAllWorkers();
                if (workers.isEmpty()) {
                    System.out.println("لا يوجد عمال متصلون حالياً.");
                    continue;
                }

                for (String workerNode : workers) {
                    byte[] data = client.getData().forPath("/service_registry/" + workerNode);
                    String address = new String(data);
                    String[] parts = address.split(":");

                    ManagedChannel channel = ManagedChannelBuilder.forAddress(parts[0], Integer.parseInt(parts[1]))
                            .usePlaintext().build();
                    SearchServiceGrpc.SearchServiceBlockingStub stub = SearchServiceGrpc.newBlockingStub(channel);

                    SearchResponse response = stub.processSearch(SearchRequest.newBuilder().setQuery(query).build());
                    System.out.println("\n[الرد من العامل " + parts[1] + "]:\n" + response.getResult());
                    channel.shutdown();
                }
            } catch (Exception e) {
                System.out.println("خطأ أثناء البحث: " + e.getMessage());
            }
        }
    }

    private static void startWorkerNode(ServiceRegistry registry) throws Exception {
        int port;
        try (ServerSocket socket = new ServerSocket(0)) { port = socket.getLocalPort(); }

        String myAddress = InetAddress.getLocalHost().getHostAddress() + ":" + port;
        registry.registerToCluster(myAddress);

        Server server = ServerBuilder.forPort(port)
                .addService(new SearchServiceImplementation())
                .build();

        System.out.println("[Worker] السيرفر جاهز كعامل على البورت: " + port);
        server.start();
        server.awaitTermination();
    }
}