package com.search.project;

import com.search.project.proto.SearchResponse;
import com.sun.net.httpserver.HttpServer;
import org.apache.curator.framework.CuratorFramework;
import java.io.OutputStream;
import java.net.InetSocketAddress;
import java.net.URLDecoder;
import java.util.List;

public class FrontendServer {
    public static void start(CuratorFramework client, ServiceRegistry registry) throws Exception {
        HttpServer server = HttpServer.create(new InetSocketAddress(9000), 0);
        server.createContext("/search", (exchange) -> {
            String queryPart = exchange.getRequestURI().getQuery();
            String searchTerm = (queryPart != null && queryPart.contains("query=")) ?
                    URLDecoder.decode(queryPart.split("query=")[1], "UTF-8") : "";

            List<SearchResponse.DocumentScore> results = SearchCoordinator.performDistributedSearch(searchTerm, client, registry);

            StringBuilder response = new StringBuilder();
            response.append("<!DOCTYPE html><html><head><meta charset='UTF-8'>");
            response.append("<title>محرك البحث الموزع</title>");

            // --- التنسيقات CSS ---
            response.append("<style>");
            response.append("body { font-family: 'Segoe UI', Tahoma, Geneva, Verdana, sans-serif; background: linear-gradient(135deg, #f5f7fa 0%, #c3cfe2 100%); min-height: 100vh; margin: 0; display: flex; flex-direction: column; align-items: center; padding-top: 50px; }");
            response.append(".container { background: white; padding: 40px; border-radius: 20px; box-shadow: 0 15px 35px rgba(0,0,0,0.1); width: 85%; max-width: 900px; text-align: center; }");
            response.append("h1 { color: #2c3e50; font-size: 2.8em; margin-bottom: 30px; letter-spacing: -1px; }");
            response.append(".search-box { display: flex; justify-content: center; gap: 10px; margin-bottom: 30px; }");
            response.append("input[type='text'] { padding: 15px 25px; width: 65%; border: 2px solid #dfe6e9; border-radius: 30px; outline: none; transition: 0.3s; font-size: 16px; }");
            response.append("input[type='text']:focus { border-color: #3498db; box-shadow: 0 0 10px rgba(52,152,219,0.2); }");
            response.append("button { padding: 15px 30px; background-color: #3498db; color: white; border: none; border-radius: 30px; cursor: pointer; font-size: 16px; font-weight: bold; transition: 0.3s; }");
            response.append("button:hover { background-color: #2980b9; transform: translateY(-2px); }");
            response.append("table { width: 100%; border-collapse: collapse; margin-top: 25px; background: white; border-radius: 12px; overflow: hidden; }");
            response.append("th { background-color: #34495e; color: white; padding: 15px; text-transform: uppercase; font-size: 13px; letter-spacing: 1px; }");
            response.append("td { padding: 15px; border-bottom: 1px solid #f1f1f1; color: #2d3436; font-size: 15px; }");
            response.append("tr:hover { background-color: #f9fbfd; }");
            response.append(".rank { font-weight: bold; color: #3498db; width: 10%; }");
            response.append(".doc-name { text-align: left; width: 65%; }");
            response.append(".score { color: #27ae60; font-weight: bold; width: 25%; }");
            response.append(".footer { margin-top: 30px; color: #636e72; font-size: 14px; }");
            response.append("</style></head>");

            // --- بنية الصفحة HTML ---
            response.append("<body>");
            response.append("<div class='container'>");
            response.append("<h1>Distributed Search Engine 🚀</h1>");

            // إعادة إضافة خانة الإدخال (هذا ما كان محذوفاً)
            response.append("<form action='/search' method='get' class='search-box'>");
            response.append("<input type='text' name='query' placeholder='أدخل كلمات البحث هنا...' value='").append(searchTerm).append("'>");
            response.append("<button type='submit'>إبحث الآن</button>");
            response.append("</form>");

            if (!searchTerm.isEmpty()) {
                response.append("<div style='text-align: left; margin-bottom: 10px; color: #7f8c8d;'>نتائج البحث عن: <b style='color: #2c3e50;'>").append(searchTerm).append("</b></div>");

                if (results.isEmpty()) {
                    response.append("<p style='color: #e74c3c;'>للأسف، لا يوجد مستندات تطابق هذا البحث.</p>");
                } else {
                    response.append("<table>");
                    response.append("<thead><tr><th>Rank</th><th>Document Name</th><th>Relevance Score (TF-IDF)</th></tr></thead>");
                    response.append("<tbody>");
                    for (int i = 0; i < results.size(); i++) {
                        response.append("<tr>");
                        response.append("<td class='rank'>#").append(i + 1).append("</td>");
                        response.append("<td class='doc-name'>").append(results.get(i).getDocName()).append("</td>");
                        response.append("<td class='score'>").append(String.format("%.6f", results.get(i).getScore())).append("</td>");
                        response.append("</tr>");
                    }
                    response.append("</tbody></table>");
                }
            }

            response.append("<div class='footer'>نظام بحث موزع يعتمد على خوارزمية TF-IDF</div>");
            response.append("</div>");
            response.append("</body></html>");

            byte[] responseBytes = response.toString().getBytes("UTF-8");
            exchange.getResponseHeaders().set("Content-Type", "text/html; charset=UTF-8");
            exchange.sendResponseHeaders(200, responseBytes.length);
            try (OutputStream os = exchange.getResponseBody()) {
                os.write(responseBytes);
            }
        });

        System.out.println("[Frontend] : http://localhost:9000/search");
        server.start();
    }
}