package AutoHealerAndClusterSearch;

import ObjectExchangeInCluster.Request;
import ObjectExchangeInCluster.Response;
import org.apache.zookeeper.KeeperException;
import org.apache.zookeeper.ZooKeeper;
import org.apache.zookeeper.data.Stat;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import java.io.*;
import java.net.ServerSocket;
import java.net.Socket;
import java.util.*;
import java.util.concurrent.*;

public class Coordinator
{
    private ZooKeeper zooKeeper;
    ///private static final String WORKERS_ZNODES_PATH = "/workers";
    private static final String PHYSICAL_ZNODES_PATH = "/physical_nodes";

    ///private static final String COORDINATOR_ZNODE_PATH = "/coordinator_node";
    private int REQUEST_RECEIVE_PORT = 12345;
    private int CLUSTER_PORT = 54321;

    private String FILES_DIRECTORY = "";
    private final Logger logger = LoggerFactory.getLogger(Coordinator.class);
    public Coordinator(ZooKeeper zooKeeper)
    {
        this.zooKeeper = zooKeeper;
        FILES_DIRECTORY = System.getProperty(("user.dir") + "/SearchFiles");
    }


    public void start() throws IOException, InterruptedException, KeeperException
    {
        ServerSocket serverSocket = new ServerSocket(REQUEST_RECEIVE_PORT);
        System.out.println("Server started on port " + REQUEST_RECEIVE_PORT);
        logger.info("Server started on port " + REQUEST_RECEIVE_PORT);

        while (true)
        {
            Socket clientSocket = serverSocket.accept();
            System.out.println("New client connected: " + clientSocket.getInetAddress());
            logger.info("New client connected: " + clientSocket.getInetAddress());

            Thread clientThread = new Thread(() ->
            {
                Response responseMap = handleClient(clientSocket);

                sendResponsesToClient(clientSocket, responseMap);
            });

            clientThread.start();
        }

    }
    private Response sendRequestToNode(Request request, String ipAddress)
    {
        Response response = null;

        try (Socket socket = new Socket(ipAddress, CLUSTER_PORT)) 
        {
          
            ObjectOutputStream objectOutputStream = new ObjectOutputStream(socket.getOutputStream());
            objectOutputStream.writeObject(request);

            // Receive the response
            ObjectInputStream objectInputStream = new ObjectInputStream(socket.getInputStream());
            response = (Response) objectInputStream.readObject();


        } catch (IOException | ClassNotFoundException e) {
            e.printStackTrace();
        }

        return response;
    }

    private void sendResponsesToClient(Socket clientSocket, Response response)
    {
        try {
            ObjectOutputStream objectOutputStream = new ObjectOutputStream(clientSocket.getOutputStream());
            /*for (Map.Entry<Serializable, Serializable> entry : responseMap.entrySet()) {
                objectOutputStream.writeObject(entry.getValue());
            }*/
            clientSocket.close();
        } catch (IOException e) {
            e.printStackTrace();
        }
    }


    private String extractRequest(Socket clientSocket) throws IOException, ClassNotFoundException
    {
        ObjectInputStream objectInputStream = new ObjectInputStream(clientSocket.getInputStream());
        String request = (String) objectInputStream.readObject();
        return request;
    }

    private Response handleClient(Socket clientSocket)
    {
        try 
        {
            String query = extractRequest(clientSocket);
            List<Response> responses = spreadQuery(query);
            processResultsAndPrepareAnswer(responses);
        }
        catch (IOException | InterruptedException | ClassNotFoundException | KeeperException e)
        {
            /**
             * retry the responses ???
            * */
            e.printStackTrace();
        }
        catch (ExecutionException e)
        {
            throw new RuntimeException(e);
        }

        return null;
    }


    /**
     * response from cluster node is tree-map where the keys are words
     * and values are list of pair<fileName,Number Of Appearance of the word in it>
     * we now must calculate the importance of each file for every word we got
     * */
    private void processResultsAndPrepareAnswer(List<Response> responses)
    {

        return;
    }

    private int countFilesInDirectory()
    {
        File directory = new File(FILES_DIRECTORY);
        File[] files = directory.listFiles();

        int fileCount = 0;
        if (files != null) {
            for (File file : files) {
                if (file.isFile()) {
                    fileCount++;
                }
            }
        }

        return fileCount;
    }


    /**
     * this method broadcasts the query for all nodes in the cluster
     * and tells them what files are every node responsible for !!
     * and try to distribute responsibility as much as possible !!!
     * then receive response from every node
     * response is tree-map here the keys are words
     * and values are list of pair<fileName,Number Of Appearance of the word in it>
     * */
    public List<Response> spreadQuery(String query) throws InterruptedException, KeeperException, IOException, ExecutionException
    {
        List<String> physicalZnodes = zooKeeper.getChildren(PHYSICAL_ZNODES_PATH, false);

        int totalFilesNumber = countFilesInDirectory();

        int filesNumberforNode = (totalFilesNumber + physicalZnodes.size()-1)/physicalZnodes.size();
        int remaining = totalFilesNumber;
        int index = 0;
        int filesOffset=0;

        ExecutorService executorService = Executors.newFixedThreadPool(physicalZnodes.size());
        List<Callable<Response>> tasks = new ArrayList<>();

        ///distributing Files for NODES!!!
        while (remaining > 0)
        {
            String physicalZnode = physicalZnodes.get(index);
            Stat stat = zooKeeper.exists(PHYSICAL_ZNODES_PATH + "/" + physicalZnodes.get(index), false);
            if (stat == null) {
                logger.warn("Physical Node : " + physicalZnode + " is dowm!");
                physicalZnodes.remove(index);
                ///nodes size ==0 ??
                filesNumberforNode = (totalFilesNumber + physicalZnodes.size() - 1) / physicalZnodes.size();
                continue;
            }
            String ipAddress = new String(
                    zooKeeper.getData(
                            PHYSICAL_ZNODES_PATH + "/" + physicalZnodes.get(index),
                                    false, stat)
            );
            Request request = new Request(query , filesNumberforNode , filesOffset);
            tasks.add(() -> sendRequestToNode(request, ipAddress));
            index = (1 + index) % physicalZnodes.size();
            filesOffset = (filesOffset + filesNumberforNode) % totalFilesNumber;
            remaining -= filesNumberforNode;
        }

        List<Future<Response>> futures = executorService.invokeAll(tasks);

        List<Response> responses = new ArrayList<>();
        for (int i = 0; i < futures.size(); i++)
        {
            Response response = futures.get(i).get();
            responses.add(response);
        }
        executorService.shutdown();

        return responses;
    }



    /*private List<String> getIPAdressesInCluster() throws InterruptedException, KeeperException
    {
        List<String> nodes = zooKeeper.getChildren(PHYSICAL_ZNODES_PATH , false);
        List<String> ipList = new ArrayList<>();
        for (String node : nodes)
        {
            String fullName = PHYSICAL_ZNODES_PATH + "/" + node;
            Stat stat = zooKeeper.exists(fullName , false);
            if (stat == null)
            {
                nodes.remove(node);
                continue;
            }
            String data = new String(zooKeeper.getData(fullName , false , stat));
            data = data.split("@")[1];
            ipList.add(data);
        }
        return ipList;
    }*/
}
