#include <mpi.h>
#include <stdlib.h>
#include <stdio.h>
#include <string.h>
#include <math.h>  
#include <stdbool.h>


void reduce_tree(
    int* send_data,
    int* recv_data,
    int count,
    MPI_Comm communicator)
{
    int my_rank;
    int com_size;
    MPI_Comm_rank(communicator, &my_rank);
    MPI_Comm_size(communicator, &com_size);

    int parent = (my_rank - 1) / 2; // Calculate the parent_rank index in the tree
    int left_rank = 2 * my_rank + 1; // Calculate left_child_rank index in the tree
    int right_rank= 2 * my_rank + 2; // Calculate the right_child_rank index in the tree
    
    if(my_rank != 0){
      recv_data = (int*)malloc(count * sizeof(int));
    }
    //initializing recv_data with zeros
    memset(recv_data, 0, count * sizeof(int));
   
    // these buffers are for storing the received data from left and right children
    int* left_data = (int*)malloc(count * sizeof(int));
    int* right_data = (int*)malloc(count * sizeof(int));

    // Receiving data from left child
    if (left_rank < com_size) {
    
        MPI_Recv(left_data, count, MPI_INT, left_rank, 0, communicator, MPI_STATUS_IGNORE);
        
        for (int i = 0; i < count; i++) {
        recv_data[i] = left_data[i] + recv_data[i];
        }   
    }

    // Receiving data from right child
    if (right_rank < com_size) {
    
        MPI_Recv(right_data, count, MPI_INT, right_rank, 0, communicator, MPI_STATUS_IGNORE);
     
        for (int i = 0; i < count; i++) {
        recv_data[i] = right_data[i] + recv_data[i];
        }
    }
   
    // adding data received from children with parent data
    for (int i = 0; i < count; i++) {
        recv_data[i] = recv_data[i] + send_data[i];
    }

    // Sending the result to the parent unless it is the root
    if (my_rank != 0) {
        MPI_Send(recv_data, count, MPI_INT, parent, 0, communicator);
        
    }

    // Free allocated memory
    free(left_data); 
    free(right_data);
}




void reduce_sequential(
    int* send_data,
    int* recv_data,
    int count,
    MPI_Comm communicator)
{
    int my_rank;
    int com_size;
    MPI_Comm_rank(communicator, &my_rank);
    MPI_Comm_size(communicator, &com_size);

    int* gather_buffer = NULL;
    if (my_rank == 0)
    {
        gather_buffer = (int*) calloc(count * com_size, sizeof(int));
    }

    MPI_Gather(send_data, count, MPI_INT, gather_buffer, count, MPI_INT, 0, communicator);

    if (my_rank == 0)
    {
        memset(recv_data, 0, count * sizeof(int));
        for (int p = 0; p < com_size; p++)
            for (int i = 0; i < count; i++)
                recv_data[i] += gather_buffer[count * p + i];
        free(gather_buffer);
    }
}



int main(int argc, char** args)
{
    MPI_Init(&argc, &args);
    int count = 10;
    int max_value = 64;
    int* recv_array_tree = NULL;
    int* recv_array_sequential = NULL;

    int my_rank;
    MPI_Comm_rank(MPI_COMM_WORLD, &my_rank);
    
    MPI_Comm communicator = MPI_COMM_WORLD;
    int com_size;
    MPI_Comm_size(communicator, &com_size);
    
    if (my_rank == 0)
    {
        recv_array_tree = (int*) malloc(count * sizeof(int));
        recv_array_sequential = (int*) malloc(count * sizeof(int));
    }

    int* send_array = (int*)malloc(count * sizeof(int));
    for (int i = 0; i < count; i++)
        send_array[i] = my_rank;
        
    MPI_Barrier(MPI_COMM_WORLD);
    double time_start = MPI_Wtime();
    reduce_tree(send_array, recv_array_tree, count, MPI_COMM_WORLD);
    double time_end = MPI_Wtime();
    double elapsed_time = time_end - time_start;
    
    MPI_Barrier(MPI_COMM_WORLD);    
    double time_start_seq = MPI_Wtime();
    reduce_sequential(send_array, recv_array_sequential, count, MPI_COMM_WORLD);
    time_end = MPI_Wtime();
    double elapsed_time_seq = time_end - time_start_seq;
    printf("\n");
    if (my_rank == 0)
    {
        printf("size of array : %d \n", count);
        printf("Number of processes: %d\n", com_size);
        printf("Processing time for reduce tree is: %lf\n", elapsed_time);
        printf("Processing time for reduce sequential is: %lf\n\n", elapsed_time_seq);
        for (int i = 0; i < count; i++)
            if (recv_array_tree[i] == recv_array_sequential[i])
                printf("At index %i: reduce_tree is %i, reduce_sequential is %i\n",
                    i, recv_array_tree[i], recv_array_sequential[i]);

        free(recv_array_tree);
        free(recv_array_sequential);
    }
    free(send_array);
    MPI_Finalize();
    return 0;

}
 
