#include <stdio.h>
#include <stdlib.h>
#include <mpi.h>
#include <cstring>
void prefix_mpi(int* block_array, int block_size, int* block_prefix, MPI_Comm communicator)
{
    int my_rank, com_size;
    MPI_Comm_rank(communicator, &my_rank);
    MPI_Comm_size(communicator, &com_size);

    int* local_prefix = (int*)malloc(block_size * sizeof(int));
    local_prefix[0] = block_array[0];
    int sum;
    // Calculate local prefix sum
    for (int i = 1; i < block_size; i++) {
        local_prefix[i] = local_prefix[i - 1] + block_array[i];
    }
    sum = local_prefix[block_size - 1];  //the total sum of each block
    
    int* prefix_sums = (int*)malloc(com_size * sizeof(int)); //contains the total sums for blocks in it
    MPI_Gather(&sum, 1, MPI_INT, prefix_sums, 1, MPI_INT, 0, communicator);
    
    //to store the values that should be added to each local_prefix
    int* accumulator= (int*)malloc(com_size * sizeof(int));
    
    if (my_rank == 0) {
        accumulator[0] = 0;
        for (int i = 1; i < com_size; i++) {
            //doing prefix_sum to accumulator
            accumulator[i] = accumulator[i-1] + prefix_sums[i-1];
        }
    }


    MPI_Bcast(accumulator, com_size, MPI_INT, 0, communicator);
    
    //block_prefix contains the final prefix sum for a clock
    for (int i = 0; i < block_size; i++) {
        block_prefix[i] = local_prefix[i] + accumulator[my_rank];
    }

    free(local_prefix);
    free(prefix_sums);
}


int main(int argc, char** argv) {
    MPI_Init(&argc, &argv); 
 
    int my_rank; 
    int com_size; 
    MPI_Comm_rank(MPI_COMM_WORLD, &my_rank); 
    MPI_Comm_size(MPI_COMM_WORLD, &com_size); 
 
    int total_array_size = 20; 
 
    if (total_array_size % com_size != 0) 
        total_array_size = (total_array_size / com_size + 1) * com_size; 
 
    int block_size = total_array_size / com_size; 
    int* total_array = NULL; 
    int* total_prefix = NULL; 
 
    if (my_rank == 0) { 
        total_array = (int*)malloc(total_array_size * sizeof(int)); 
        total_prefix = (int*)malloc(total_array_size * sizeof(int)); 
        for (int i = 0; i < total_array_size; i++) 
            total_array[i] = rand() % 11; 
        printf("Total Array: ");
        for (int i = 0; i < total_array_size; i++)
            printf("%d ", total_array[i]);
        printf("\n\n");
    } 
 
    int* block_array = (int*)malloc(block_size * sizeof(int)); 
    int* block_prefix = (int*)malloc(block_size * sizeof(int)); 
 
    MPI_Scatter(total_array, block_size, MPI_INT, 
        block_array, block_size, MPI_INT, 0, MPI_COMM_WORLD); 
 
    prefix_mpi(block_array, block_size, block_prefix, MPI_COMM_WORLD); 
 
    MPI_Gather(block_prefix, block_size, MPI_INT, 
        total_prefix, block_size, MPI_INT, 0, MPI_COMM_WORLD); 
 
    int accum = 0; 
    if (my_rank == 0) { 
        for (int i = 1; i < total_array_size; i++) { 
            accum += total_array[i - 1]; 
            if (total_prefix[i-1] != accum) 
                printf("Error at index %i: %i expected, %i computed\n", i, accum, total_prefix[i]); 
        } 
        printf("Prefix sum for the array is :\n");
        for (int i = 1; i < total_array_size; i++) {  
            printf("%d ",total_prefix[i]); 
        }
        printf("\n\n");
        printf("Test completed!\n");  
        free(total_array); 
        free(total_prefix); 
    } 
    
    free(block_array); 
    free(block_prefix); 
 
    MPI_Finalize(); 
     
    return 0; 
}
