#include <mpi.h>
#include <stdio.h>
#include <stdlib.h>
#include <time.h>

// Perform tree-based reduction on local_data using addition as the operation
int reduce_tree(int local_data, int rank, int size, MPI_Comm comm) {
    int step = 1;

    printf("[+] Process %d starting with local data: %d\n", rank, local_data);

    while (step < size) {
        int partner = rank ^ step;  // XOR operation to find partner

        if (partner < size) {
            int recv_data = 0;

            // Exchange data with the partner
            MPI_Sendrecv(&local_data, 1, MPI_INT, partner, 0, 
                         &recv_data, 1, MPI_INT, partner, 0, 
                         comm, MPI_STATUS_IGNORE);

            printf("[+] Process %d received data from process %d: %d\n", rank, partner, recv_data);

            // Combine received data with the local data
            local_data += recv_data;

            printf("[+] Process %d updated local data: %d\n", rank, local_data);
        }

        // Move to the next step
        step *= 2;
    }

    // Process 0 returns the final result
    if (rank == 0) {
        return local_data;
    } else {
        return 0;
    }
}

int main(int argc, char** argv) {
    MPI_Init(&argc, &argv);

    int rank, size;
    MPI_Comm comm = MPI_COMM_WORLD;
    MPI_Comm_rank(comm, &rank);
    MPI_Comm_size(comm, &size);

    // Initialize local data for each process
    int local_data = rank + 1;

    // Start timer
    double start_time = MPI_Wtime();

    // Perform the tree-based reduce operation
    int result = reduce_tree(local_data, rank, size, comm);

    // End timer
    double end_time = MPI_Wtime();

    // Elapsed time
    double elapsed_time = end_time - start_time;

    if (rank == 0) {
        printf("[+] Final result after reduction: %d\n", result);
        printf("[+] Time taken for the reduction: %.6f seconds\n", elapsed_time);
    }

    MPI_Finalize();
    return 0;
}
