package com.quiz.server;
import com.quiz.common.*;
import java.rmi.RemoteException;
import java.rmi.server.UnicastRemoteObject;
import java.rmi.registry.LocateRegistry;
import java.rmi.registry.Registry;
import java.util.*;

public class QuizServer extends UnicastRemoteObject implements QuizService {

    private Map<String, StudentScore> studentScores = new HashMap<>();
    private List<Question> questions = new ArrayList<>();
    private Map<String, Question> currentQuestions = new HashMap<>();
    private Map<String, QuizCallback> callbacks = new HashMap<>();

    protected QuizServer() throws RemoteException {
        super();
        questions.add(new Question("What is 2 + 2?", "4"));
        questions.add(new Question("Capital of France?", "Paris"));
        questions.add(new Question("Java is ___?", "Programming language"));
        // don't forget to add more questions in the future
    }

    @Override
    public Question getQuestion(String studentName) throws RemoteException {
        studentScores.putIfAbsent(studentName, new StudentScore(studentName));
        Question q = questions.get(new Random().nextInt(questions.size()));
        currentQuestions.put(studentName, q);
        return q;
    }

    @Override
    public String submitAnswer(String studentName, String answer) throws RemoteException {
        Question q = currentQuestions.get(studentName);
        if (q == null) return "No question assigned!";
        String feedback;
        if (q.getAnswer().equalsIgnoreCase(answer.trim())) {
            studentScores.get(studentName).incrementScore();
            feedback = "Correct! Score: " + studentScores.get(studentName).getScore();
        } else {
            feedback = "Wrong! Score: " + studentScores.get(studentName).getScore();
        }
        updateLeaderboard();
        return feedback;
    }

    @Override
    public void registerCallback(String studentName, QuizCallback callback) throws RemoteException {
        callbacks.put(studentName, callback);
    }

    private void updateLeaderboard() {
        List<StudentScore> top = new ArrayList<>(studentScores.values());
        top.sort((a, b) -> b.getScore() - a.getScore());
        StringBuilder leaderboard = new StringBuilder("Leaderboard:\n");
        for (int i = 0; i < Math.min(3, top.size()); i++) {
            leaderboard.append((i+1) + ". " + top.get(i).getName() + " - " + top.get(i).getScore() + "\n");
        }
        for (QuizCallback cb : callbacks.values()) {
            try { cb.updateLeaderboard(leaderboard.toString()); }
            catch (Exception ignored) {}
        }
    }

    public static void main(String[] args) {
        try {
            QuizServer server = new QuizServer();
            Registry registry = LocateRegistry.createRegistry(1099);
            registry.rebind("QuizService", server);
            System.out.println("Quiz Server is running...");
            synchronized (server) {
            server.wait();
            }
        } catch (Exception e) {
            e.printStackTrace();
        }
    }
}
