import grpc from "@grpc/grpc-js";
import protoLoader from "@grpc/proto-loader";
import sqlite3 from "sqlite3";
import { open } from "sqlite";

// Initialize SQLite
let db;

async function initDb() {
  db = await open({
    filename: "./logs.db",
    driver: sqlite3.Database,
  });

  await db.exec(`
    CREATE TABLE IF NOT EXISTS multiply_logs (
      id INTEGER PRIMARY KEY AUTOINCREMENT,
      a REAL,
      b REAL,
      result REAL,
      timestamp DATETIME DEFAULT CURRENT_TIMESTAMP
    )
  `);

  console.log("SQLite database initialized and table ready.");
}

const packageDefinition = protoLoader.loadSync("./multiply.proto", {});
const multiplyProto = grpc.loadPackageDefinition(packageDefinition).multiplyservice;

// Multiply RPC
async function multiply(call, callback) {
  const { a, b } = call.request;
  const result = a * b;

  try {
    await db.run("INSERT INTO multiply_logs (a, b, result) VALUES (?, ?, ?)", [a, b, result]);
    console.log(`Logged: ${a} * ${b} = ${result}`);
  } catch (err) {
    console.error("Database logging failed:", err);
  }

  callback(null, { result });
}

// NEW: StreamLogs RPC
async function streamLogs(call) {
  const limit = call.request.limit || 10; // default 10 logs
  try {
    const rows = await db.all(
      "SELECT a, b, result, timestamp FROM multiply_logs ORDER BY id DESC LIMIT ?",
      [limit]
    );

    for (const row of rows) {
      call.write({
        a: row.a,
        b: row.b,
        result: row.result,
        timestamp: row.timestamp,
      });
    }
    call.end();
  } catch (err) {
    console.error("Streaming logs failed:", err);
    call.end();
  }
}

// Start the gRPC server
async function main() {
  await initDb();

  const server = new grpc.Server();
  server.addService(multiplyProto.MultiplyService.service, {
    Multiply: multiply,
    StreamLogs: streamLogs, 
  });

  const port = "50052";
  server.bindAsync(`0.0.0.0:${port}`, grpc.ServerCredentials.createInsecure(), () => {
    server.start();
    console.log(`Multiply gRPC Server running on port ${port}`);
  });
}

main();

