package org.example;

import java.util.*;
import java.util.concurrent.*;

public class Stemming {

    // 1. Shared Blocking Queues
    // Queue 1: To transfer raw lines from the Producer to the Multiple Consumers
    private static final BlockingQueue<String> queue1_Lines = new LinkedBlockingQueue<>(100);
    // Queue 2: To transfer partial results  from Consumers to the Single Writer
    private static final BlockingQueue<List<String>> queue2_PartialResults = new LinkedBlockingQueue<>();
    // Special signal to indicate End-Of-File (EOF)
    private static final String EOF = "EOF_SIGNAL_XYZ";
    public static final int NUM_CONSUMERS = 4; // Number of parallel processing threads

    public static void main(String[] args) {
        // Define file paths
        String inputPath = "input.txt";
        String outputPath = "output.txt";  // Will be created in the working directory

        // 1. Create Threads
        // The Producer reads the file and puts lines into queue1
        Thread producer = new Thread(new Producer(inputPath, queue1_Lines, EOF), "Producer");
        // The Writer collects all partial maps from queue2 and writes the final output
        Thread writer = new Thread(new SingleConsumerWriter(outputPath, queue2_PartialResults, NUM_CONSUMERS), "WriterConsumer");

        // 2. Create and start Multiple Consumers
        // Uses a fixed thread pool to manage the worker threads
        ExecutorService consumerPool = Executors.newFixedThreadPool(NUM_CONSUMERS);
        for (int i = 0; i < NUM_CONSUMERS; i++) {
            // Each MultipleConsumer thread takes lines from queue1 and puts partial results into queue2
            consumerPool.execute(new MultipleConsumers(queue1_Lines, queue2_PartialResults, EOF));
        }

        // 3. Start all threads
        producer.start();
        writer.start();

        // 4. Shutdown services and wait for termination
        try {
            // Wait for the Producer to finish reading the entire file
            producer.join();

            // Shut down the consumer pool and wait for all tasks to complete (since EOF signals were sent)
            consumerPool.shutdown();
            consumerPool.awaitTermination(1, TimeUnit.MINUTES);

            // Wait for the Writer to finish combining results and writing the output file
            writer.join();

            System.out.println(" Processing completed successfully. Check file: " + outputPath);
        } catch (InterruptedException e) {
            Thread.currentThread().interrupt();
            System.err.println(" Execution process was interrupted.");
        }
    }


}