package org.example;

import org.apache.zookeeper.*;
import org.apache.zookeeper.data.Stat;

import java.io.IOException;
import java.util.Collections;
import java.util.List;

public class LeaderElectionImprovement implements Watcher {

    // **1. Connection Settings**
    private static final String address = "192.168.144.134:2181";
    private static final int SESSION_TIMEOUT = 3000;
    private static final String ELECTION_NAMESPACE = "/election";

    private String currentZnodeName;
    private ZooKeeper zooKeeper;

    public static void main(String[] args) throws IOException, InterruptedException, KeeperException {
        // Class Instance
        LeaderElectionImprovement leaderElectionImprovement = new LeaderElectionImprovement();

        leaderElectionImprovement.connectToZookeeper();
        leaderElectionImprovement.volunteerForLeadership();
        leaderElectionImprovement.electLeader();
        leaderElectionImprovement.run();
        leaderElectionImprovement.close();
        System.out.println("Successfully Closed");
    }

    public void volunteerForLeadership() throws InterruptedException, KeeperException {
        String znodePrefix = ELECTION_NAMESPACE + "/c_";

        // Creating an Ephemeral Sequential znode
        String znodeFullPath = zooKeeper.create(znodePrefix, new byte[]{}, ZooDefs.Ids.OPEN_ACL_UNSAFE, CreateMode.EPHEMERAL_SEQUENTIAL);

        System.out.println(" Successfully volunteered. Full Znode path: " + znodeFullPath);
        this.currentZnodeName = znodeFullPath.replace(ELECTION_NAMESPACE + "/", "");
    }

    public void electLeader() throws InterruptedException, KeeperException {

        // Getting children without setting a watcher on the parent node (false)
        List<String> children = zooKeeper.getChildren(ELECTION_NAMESPACE, false);

        Collections.sort(children); // Sorting the children

        String smallestChild = children.get(0);
        int index = children.indexOf(currentZnodeName);

        if (index == 0) {
            // I am the Leader!
            System.out.println(" I am the Leader!");
        } else {
            // I am a Follower. I must watch the immediately preceding rival.
            String smallerZnodeName = children.get(index - 1);
            String watcherZnodePath = ELECTION_NAMESPACE + "/" + smallerZnodeName;

            System.out.println(" I am a Follower.");
            System.out.println(" I will watch the closest preceding rival: " + smallerZnodeName);

            // Using exists() to set a NodeDeleted Watcher on the preceding node only
            Stat stat = zooKeeper.exists(watcherZnodePath, this);

            if (stat == null) {
                // If the previous znode is not found, it means it was deleted before we set the watcher.
                System.out.println(" Failure! Preceding znode already deleted. Re-electing immediately.");
                electLeader();
            }
        }
    }

    private void close() throws InterruptedException {
        this.zooKeeper.close();
    }

    public void connectToZookeeper() throws IOException {
        this.zooKeeper = new ZooKeeper(address, SESSION_TIMEOUT, this);
    }

    public void run() throws InterruptedException {
        // Keeps the main thread waiting to process Watcher events
        synchronized (zooKeeper) {
            zooKeeper.wait();
        }
    }


    @Override
    public void process(WatchedEvent watchedEvent) {
        // Connection and Disconnection handling
        if (watchedEvent.getType() == Event.EventType.None) {
            if (watchedEvent.getState() == Event.KeeperState.SyncConnected) {
                System.out.println(" Successfully connected to Zookeeper");
            } else if (watchedEvent.getState() == Event.KeeperState.Disconnected) {
                synchronized (zooKeeper) {
                    System.out.println(" Disconnected from Zookeeper");
                    zooKeeper.notifyAll();
                }
            }
        }

        // **Handling the Deletion Event**
        if (watchedEvent.getType() == Event.EventType.NodeDeleted) {
            // This is triggered when the znode we are watching (our preceding rival) is deleted
            try {
                System.out.println(" Notification! Rival we were watching has been deleted: " + watchedEvent.getPath());
                electLeader(); // Re-elect to determine if we are the new leader
            } catch (InterruptedException | KeeperException e) {
                e.printStackTrace();
            }
        }
    }
}