package org.example;

// --- هذه هي السطور المهمة التي تحل المشكلة ---
import com.example.book.Book;
import com.google.protobuf.InvalidProtocolBufferException;
import java.util.Arrays;
import java.nio.charset.StandardCharsets;
// --- نهاية السطور المهمة ---


public class Main {

    public static void main(String[] args) {

        // --- الخطوة 1: إنشاء كائن (Instance) من رسالة Book ---

        // 1a. بناء الرسالة المتداخلة (Nested Message) أولاً
        Book.Publisher publisher = Book.Publisher.newBuilder()
                .setName("O'Reilly")
                .setLocation("USA")
                .build();

        // 1b. بناء الرسالة الرئيسية (Book)
        Book myBook = Book.newBuilder()
                .setTitle("gRPC: The Definitive Guide")
                .setAuthor("Ray")
                .setIsbn("123-456-789")
                .setPublicationYear(2025)
                .setGenre(Book.Genre.SCIENCE) // استخدام الـ Enum
                .setPublisher(publisher)      // استخدام الـ Nested Message
                .build();

        System.out.println("--- 1. الكائن الأصلي (قبل الترميز) ---");
        System.out.println(myBook);


        // --- الخطوة 2: ترميز الكائن إلى مصفوفة بايتات (Serialization) ---

        byte[] bookAsBytes = myBook.toByteArray();

        System.out.println("\n--- 2. عملية الترميز (Encode) ---");
        System.out.println("تم الترميز إلى " + bookAsBytes.length + " بايت");
        System.out.println("شكل البايتات: " + Arrays.toString(bookAsBytes));


        // --- الخطوة 3: فك ترميز البايتات رجوعاً إلى كائن (Deserialization) ---

        try {
            // نستخدم parseFrom لتحويل المصفوفة إلى كائن Book
            Book decodedBook = Book.parseFrom(bookAsBytes);

            System.out.println("\n--- 3. الكائن (بعد فك الترميز) ---");
            System.out.println(decodedBook);

            // --- 4. التحقق من أن البيانات متطابقة ---
            System.out.println("\n--- 4. التحقق من البيانات ---");
            System.out.println("هل الكائن الأصلي يساوي الكائن المترجم؟ " + myBook.equals(decodedBook));
            System.out.println("اسم الناشر (من الكائن المترجم): " + decodedBook.getPublisher().getName());

        } catch (InvalidProtocolBufferException e) {
            System.err.println("حدث خطأ أثناء فك الترميز: " + e.getMessage());
        }
        // --- الخطوة 4: (الطلب الجديد) المقارنة مع JSON ---
        // (الطلب الثالث - مقارنة الحجم)

        String bookAsJson = """
        {
          "title": "gRPC: The Definitive Guide",
          "author": "Ray",
          "isbn": "123-456-789",
          "publication_year": 2025,
          "genre": "SCIENCE",
          "publisher": {
            "name": "O'Reilly",
            "location": "USA"
          }
        }
        """;

        int jsonSize = bookAsJson.getBytes(StandardCharsets.UTF_8).length;
        int protoSize = bookAsBytes.length; // الحجم من الخطوة 2

        System.out.println("\n--- 4. مقارنة حجم Protobuf مع JSON ---");
        System.out.println("حجم البيانات كـ Protobuf: " + protoSize + " bytes");
        System.out.println("حجم البيانات كـ JSON:      " + jsonSize + " bytes");

        double saving = 100.0 * (jsonSize - protoSize) / jsonSize;
        System.out.println("\nالنتيجة:");
        System.out.println("Protobuf أصغر بـ " + (jsonSize - protoSize) + " بايت.");
        System.out.printf("أي أن Protobuf يوفر حوالي %.1f%% من حجم البيانات مقارنة بـ JSON.\n", saving);

    }
}