package com.distributed.search;

import org.apache.curator.framework.CuratorFramework;
import org.apache.curator.framework.CuratorFrameworkFactory;
import org.apache.curator.retry.ExponentialBackoffRetry;

/**
 * النسخة النهائية المدمجة مع واجهة FrontEnd ونظام التنسيق الموزع
 */
public class SearchApplication {
    // عنوان سيرفر ZooKeeper في المعهد
    private static final String ZK_ADDRESS = "172.29.3.101:2181";
    // المجلد المحلي للبيانات
    private static final String DATA_DIR = "D:\\search_data";

    // المنافذ الثابتة للقائد
    private static final int FRONT_END_PORT = 8080; // للمتصفح
    private static final int COORDINATOR_PORT = 9090; // للتواصل بين الفرونت والقائد

    public static void main(String[] args) {
        try {
            System.out.println("[System] Initializing connection to remote ZooKeeper: " + ZK_ADDRESS);

            // 1.create client
            CuratorFramework client = CuratorFrameworkFactory.builder()
                    .connectString(ZK_ADDRESS)
                    .retryPolicy(new ExponentialBackoffRetry(1000, 3))
                    .sessionTimeoutMs(15000)
                    .connectionTimeoutMs(10000)
                    .build();

            client.start();
            client.blockUntilConnected();
            System.out.println("[System] Successfully connected to ZooKeeper.");

            //2.Incial registery
            ServiceRegistry registry = new ServiceRegistry(client);
            SearchEngine engine = new SearchEngine(DATA_DIR);

            // 3. (WorkerServer) for each node
            WorkerServer workerServer = new WorkerServer(engine);
            int myPort = workerServer.getPort();

            // 4.register worker |(active)
            registry.registerWorker(myPort);

            // 5. start leader election
            LeaderElectionManager electionManager = new LeaderElectionManager(client, () -> {
                try {
                    System.out.println("\n[Leader] *** I am the new Leader of the cluster! ***");

                    // turn coordinator
                    Coordinator coordinator = new Coordinator(registry, DATA_DIR, COORDINATOR_PORT);

                    //turn frontEndServer on browser
                    FrontEndServer frontEnd = new FrontEndServer(FRONT_END_PORT, registry);
                    frontEnd.start();

                    System.out.println("[Leader] FrontEnd is live at http://localhost:" + FRONT_END_PORT);

                    //
                    coordinator.start();

                } catch (Exception e) {
                    System.err.println("[Critical] Leader initialization failed: " + e.getMessage());
                    e.printStackTrace();
                }
            });

            System.out.println("[System] Entering Leader Election queue...");
            electionManager.start();

            // 6. إبقاء العقدة تعمل لمراقبة حالة المجموعة
            System.out.println("[System] Node is active and waiting. Monitoring cluster status...");

            while (true) {
                Thread.sleep(10000);
            }

        } catch (Exception e) {
            System.err.println("[System] Application Error: " + e.getMessage());
            e.printStackTrace();
            System.exit(1);
        }
    }
}