package com.distributed.search;

import com.distributed.search.model.SearchTask;
import com.distributed.search.model.TaskResult;
import java.io.*;
import java.net.ServerSocket;
import java.net.Socket;
import java.nio.file.Files;
import java.nio.file.Path;
import java.nio.file.Paths;
import java.util.*;
import java.util.stream.Collectors;
import java.util.stream.Stream;

public class Coordinator {
    private final ServiceRegistry registry;
    private final String dataDirectory;
    private final int coordinatorPort; // منفذ خاص لاستقبال طلبات الـ Front-End

    public Coordinator(ServiceRegistry registry, String dataDirectory, int coordinatorPort) {
        this.registry = registry;
        this.dataDirectory = dataDirectory;
        this.coordinatorPort = coordinatorPort;
    }

    public void start() throws Exception {
        // 1. تسجيل عنوان القائد في ZooKeeper لكي يراه الـ Front-End
        registry.registerLeader(coordinatorPort);

        System.out.println("[Coordinator] I am the leader. Listening for Front-End on port: " + coordinatorPort);

        // 2. فتح سيرفر للاستماع لطلبات الـ Front-End
        try (ServerSocket serverSocket = new ServerSocket(coordinatorPort)) {
            while (true) {
                try (Socket frontEndSocket = serverSocket.accept();
                     BufferedReader reader = new BufferedReader(new InputStreamReader(frontEndSocket.getInputStream()));
                     PrintWriter writer = new PrintWriter(frontEndSocket.getOutputStream(), true)) {

                    // استقبال كلمة البحث من الـ Front-End
                    String query = reader.readLine();
                    if (query != null && !query.isEmpty()) {
                        System.out.println("[Coordinator] Received web query: " + query);

                        // تنفيذ البحث الموزع والحصول على النتائج كـ String
                        String results = performSearch(query);

                        // إرسال النتائج النهائية للـ Front-End ليتم عرضها في المتصفح
                        writer.println(results);
                    }
                } catch (Exception e) {
                    System.err.println("[Coordinator] Error handling Front-End request: " + e.getMessage());
                }
            }
        }
    }

    // قمنا بتعديل الدالة لتعيد String يحتوي على النتائج بدلاً من الطباعة فقط
    public String performSearch(String query) throws Exception {
        List<String> allWorkers = registry.getActiveWorkers();
        int myPort = WorkerServer.getLocalBoundPort();

        List<String> externalWorkers = allWorkers.stream()
                .filter(worker -> !worker.contains(":" + myPort))
                .collect(Collectors.toList());

        List<String> allFiles = scanFiles();

        if (externalWorkers.isEmpty()) return "Error: No external workers available.";
        if (allFiles.isEmpty()) return "Error: No files in directory.";

        Map<String, Double> globalTfResults = new HashMap<>();
        int filesPerWorker = (int) Math.ceil((double) allFiles.size() / externalWorkers.size());

        for (int i = 0; i < externalWorkers.size(); i++) {
            int start = i * filesPerWorker;
            int end = Math.min(start + filesPerWorker, allFiles.size());
            if (start >= end) break;

            List<String> workerFiles = allFiles.subList(start, end);
            String nodeData = externalWorkers.get(i).replace("worker_", "");
            String[] addr = nodeData.split(":");

            globalTfResults.putAll(sendTask(addr[0], Integer.parseInt(addr[1]), query, workerFiles));
        }

        Map<String, Double> finalTfIdfResults = calculateFinalScores(globalTfResults, allFiles.size());

        // تحويل النتائج المرتبة إلى نص لإرساله للـ Front-End
        return formatResults(finalTfIdfResults);
    }

    private String formatResults(Map<String, Double> results) {
        StringBuilder sb = new StringBuilder();
        List<Map.Entry<String, Double>> sorted = results.entrySet().stream()
                .filter(e -> e.getValue() > 0)
                .sorted(Map.Entry.<String, Double>comparingByValue().reversed())
                .limit(10)
                .collect(Collectors.toList());

        if (sorted.isEmpty()) return "No results found for this term.";

        for (Map.Entry<String, Double> entry : sorted) {
            sb.append(String.format("File: %s | Score: %.6f\n", entry.getKey(), entry.getValue()));
        }
        return sb.toString();
    }

    private Map<String, Double> calculateFinalScores(Map<String, Double> tfResults, int totalDocsCount) {
        Map<String, Double> tfIdfResults = new HashMap<>();
        long docsWithTermCount = tfResults.values().stream().filter(score -> score > 0).count();
        if (docsWithTermCount == 0) return tfIdfResults;

        double idf = Math.log10((double) totalDocsCount / docsWithTermCount);
        for (Map.Entry<String, Double> entry : tfResults.entrySet()) {
            tfIdfResults.put(entry.getKey(), entry.getValue() * idf);
        }
        return tfIdfResults;
    }

    private List<String> scanFiles() throws IOException {
        Path path = Paths.get(dataDirectory);
        if (!Files.exists(path)) return Collections.emptyList();
        try (Stream<Path> stream = Files.list(path)) {
            return stream.filter(Files::isRegularFile)
                    .map(p -> p.getFileName().toString())
                    .filter(name -> name.endsWith(".txt"))
                    .collect(Collectors.toList());
        }
    }

    private Map<String, Double> sendTask(String host, int port, String query, List<String> files) {
        try (Socket socket = new Socket(host, port)) {
            socket.setSoTimeout(5000);
            SearchTask.newBuilder().setQuery(query).addAllFilePaths(files).build()
                    .writeDelimitedTo(socket.getOutputStream());
            TaskResult result = TaskResult.parseDelimitedFrom(socket.getInputStream());
            return result != null ? result.getDocTfScoresMap() : Collections.emptyMap();
        } catch (Exception e) {
            return Collections.emptyMap();
        }
    }
}