package com.distributed.search;

import com.sun.net.httpserver.HttpServer;
import com.sun.net.httpserver.HttpHandler;
import com.sun.net.httpserver.HttpExchange;
import java.io.*;
import java.net.InetSocketAddress;
import java.net.Socket;
import java.nio.charset.StandardCharsets;

public class FrontEndServer {
    private final int port;
    private final ServiceRegistry registry;

    // نمط CSS مشترك لجعل الواجهة تبدو احترافية
    private static final String CSS =
            "<style>" +
                    "body { font-family: 'Segoe UI', Tahoma, Geneva, Verdana, sans-serif; background-color: #f8f9fa; display: flex; flex-direction: column; align-items: center; padding-top: 50px; color: #333; }" +
                    ".container { background: white; padding: 30px; border-radius: 12px; box-shadow: 0 4px 6px rgba(0,0,0,0.1); width: 80%; max-width: 600px; text-align: center; }" +
                    "h1 { color: #4285F4; margin-bottom: 30px; font-size: 2.5em; }" +
                    "input[type='text'] { width: 70%; padding: 12px 20px; margin: 8px 0; border: 1px solid #dfe1e5; border-radius: 24px; outline: none; font-size: 16px; transition: 0.3s; }" +
                    "input[type='text']:focus { box-shadow: 0 1px 6px rgba(32,33,36,0.28); border-color: rgba(223,225,229,0); }" +
                    "button { background-color: #f8f9fa; border: 1px solid #f8f9fa; border-radius: 4px; color: #3c4043; font-size: 14px; margin: 11px 4px; padding: 10px 24px; cursor: pointer; transition: 0.2s; }" +
                    "button:hover { border: 1px solid #dadce0; box-shadow: 0 1px 1px rgba(0,0,0,0.1); color: #202124; }" +
                    ".results-container { text-align: left; width: 80%; max-width: 800px; margin-top: 20px; }" +
                    ".result-item { background: white; padding: 15px; border-radius: 8px; margin-bottom: 10px; border-left: 5px solid #4285F4; box-shadow: 0 2px 4px rgba(0,0,0,0.05); }" +
                    ".file-name { font-weight: bold; color: #1a0dab; font-size: 18px; text-decoration: none; }" +
                    ".score { color: #006621; font-size: 14px; margin-top: 5px; }" +
                    ".no-results { color: #70757a; font-style: italic; }" +
                    "a.back-link { margin-top: 20px; text-decoration: none; color: #4285F4; font-weight: bold; }" +
                    "</style>";

    public FrontEndServer(int port, ServiceRegistry registry) {
        this.port = port;
        this.registry = registry;
    }

    public void start() throws IOException {
        HttpServer server = HttpServer.create(new InetSocketAddress(port), 0);
        server.createContext("/search", new SearchHandler());
        server.createContext("/", new HomeHandler());
        server.setExecutor(null);
        System.out.println("[FrontEnd] Modern Web interface ready at http://localhost:" + port + "/");
        server.start();
    }

    private class HomeHandler implements HttpHandler {
        @Override
        public void handle(HttpExchange exchange) throws IOException {
            String html = "<html><head><title>Distributed Search</title>" + CSS + "</head>" +
                    "<body>" +
                    "<div class='container'>" +
                    "<h1>🔍 Distributed Search</h1>" +
                    "<form action='/search' method='get'>" +
                    "<input type='text' name='query' placeholder='What are you looking for?' required autofocus>" +
                    "<br><button type='submit'>Cluster Search</button>" +
                    "</form>" +
                    "</div>" +
                    "</body></html>";
            sendResponse(exchange, html);
        }
    }

    private class SearchHandler implements HttpHandler {
        @Override
        public void handle(HttpExchange exchange) throws IOException {
            String uri = exchange.getRequestURI().toString();
            String query = "";

            if (uri.contains("query=")) {
                query = uri.substring(uri.indexOf("query=") + 6);
                query = java.net.URLDecoder.decode(query, StandardCharsets.UTF_8);
            }

            String rawResults;
            try {
                String leaderAddress = registry.getLeaderAddress();
                rawResults = (leaderAddress == null) ? "Error: No Leader found." : contactLeader(leaderAddress, query);
            } catch (Exception e) {
                rawResults = "System Error: " + e.getMessage();
            }

            // تحويل النتائج الخام إلى بطاقات HTML
            StringBuilder resultsHtml = new StringBuilder();
            if (rawResults.trim().isEmpty() || rawResults.contains("No results")) {
                resultsHtml.append("<p class='no-results'>No documents matched your query.</p>");
            } else {
                for (String line : rawResults.split("\n")) {
                    if (line.contains("|")) {
                        String[] parts = line.split("\\|");
                        resultsHtml.append("<div class='result-item'>")
                                .append("<div class='file-name'>📄 ").append(parts[0].replace("File:", "").trim()).append("</div>")
                                .append("<div class='score'>").append(parts[1].trim()).append("</div>")
                                .append("</div>");
                    }
                }
            }

            String finalHtml = "<html><head><title>Search Results</title>" + CSS + "</head>" +
                    "<body>" +
                    "<div class='results-container'>" +
                    "<h2>Results for: <span style='color:#4285F4'>" + query + "</span></h2>" +
                    resultsHtml.toString() +
                    "<br><a href='/' class='back-link'>← Back to Search</a>" +
                    "</div>" +
                    "</body></html>";

            sendResponse(exchange, finalHtml);
        }
    }

    private String contactLeader(String leaderAddr, String query) {
        String[] parts = leaderAddr.split(":");
        try (Socket socket = new Socket(parts[0], Integer.parseInt(parts[1]));
             PrintWriter writer = new PrintWriter(socket.getOutputStream(), true);
             BufferedReader reader = new BufferedReader(new InputStreamReader(socket.getInputStream()))) {
            writer.println(query);
            StringBuilder response = new StringBuilder();
            String line;
            while ((line = reader.readLine()) != null) response.append(line).append("\n");
            return response.toString();
        } catch (IOException e) {
            return "Failed to connect to Leader.";
        }
    }

    private void sendResponse(HttpExchange exchange, String response) throws IOException {
        byte[] bytes = response.getBytes(StandardCharsets.UTF_8);
        exchange.getResponseHeaders().set("Content-Type", "text/html; charset=UTF-8");
        exchange.sendResponseHeaders(200, bytes.length);
        try (OutputStream os = exchange.getResponseBody()) {
            os.write(bytes);
        }
    }
}