package com.distributed.search;

import com.distributed.search.model.SearchTask;
import com.distributed.search.model.TaskResult;
import java.io.IOException;
import java.net.Socket;
import java.nio.file.Files;
import java.nio.file.Path;
import java.nio.file.Paths;
import java.util.*;
import java.util.stream.Collectors;
import java.util.stream.Stream;

public class Coordinator {
    private final ServiceRegistry registry;
    private final String dataDirectory;

    public Coordinator(ServiceRegistry registry, String dataDirectory) {
        this.registry = registry;
        this.dataDirectory = dataDirectory;
    }

    public void start() throws Exception {
        Scanner scanner = new Scanner(System.in);
        while (true) {
            System.out.println("\n[Coordinator] Enter search term (or 'exit'):");
            if (!scanner.hasNextLine()) break;
            String query = scanner.nextLine();
            if (query.equalsIgnoreCase("exit")) break;

            List<String> workers = registry.getActiveWorkers();
            List<String> allFiles = scanFiles();

            if (workers.isEmpty()) {
                System.out.println("[Coordinator] No workers available!");
                continue;
            }
            if (allFiles.isEmpty()) {
                System.out.println("[Coordinator] No files found in directory: " + dataDirectory);
                continue;
            }

            Map<String, Double> globalResults = new HashMap<>();
            int filesPerWorker = (int) Math.ceil((double) allFiles.size() / workers.size());

            for (int i = 0; i < workers.size(); i++) {
                int start = i * filesPerWorker;
                int end = Math.min(start + filesPerWorker, allFiles.size());
                if (start >= end) break;

                List<String> workerFiles = allFiles.subList(start, end);

                // Safe parsing of worker info from ZooKeeper
                String nodeName = workers.get(i);
                String nodeData = nodeName.replace("worker_", "");

                if (!nodeData.contains(":")) {
                    System.err.println("[Coordinator] Skipping invalid worker node: " + nodeName);
                    continue;
                }

                try {
                    String[] addrParts = nodeData.split(":");
                    String host = addrParts[0];
                    int port = Integer.parseInt(addrParts[1]);

                    globalResults.putAll(sendTask(host, port, query, workerFiles));
                } catch (Exception e) {
                    System.err.println("[Coordinator] Failed to parse worker address: " + nodeData);
                }
            }
            printRankedResults(globalResults);
        }
    }

    private List<String> scanFiles() throws IOException {
        Path path = Paths.get(dataDirectory);
        if (!Files.exists(path)) return Collections.emptyList();

        try (Stream<Path> stream = Files.list(path)) {
            return stream.filter(Files::isRegularFile)
                    .map(p -> p.getFileName().toString())
                    .filter(name -> name.endsWith(".txt"))
                    .collect(Collectors.toList());
        }
    }

    private Map<String, Double> sendTask(String host, int port, String query, List<String> files) {
        try (Socket socket = new Socket(host, port)) {
            socket.setSoTimeout(5000); // 5 seconds timeout
            SearchTask.newBuilder().setQuery(query).addAllFilePaths(files).build()
                    .writeDelimitedTo(socket.getOutputStream());

            TaskResult result = TaskResult.parseDelimitedFrom(socket.getInputStream());
            return result != null ? result.getDocTfScoresMap() : Collections.emptyMap();
        } catch (Exception e) {
            System.err.println("[Coordinator] Error communicating with worker " + host + ":" + port);
            return Collections.emptyMap();
        }
    }

    private void printRankedResults(Map<String, Double> results) {
        System.out.println("\n--- Final Ranked Search Results ---");
        List<Map.Entry<String, Double>> ranked = results.entrySet().stream()
                .filter(e -> e.getValue() > 0)
                .sorted(Map.Entry.<String, Double>comparingByValue().reversed())
                .collect(Collectors.toList());

        if (ranked.isEmpty()) {
            System.out.println("No matching results found.");
        } else {
            ranked.forEach(e -> System.out.printf("File: %-12s | Score: %.4f%n", e.getKey(), e.getValue()));
        }
    }
}