package com.distributed.search;

import org.apache.curator.framework.CuratorFramework;
import org.apache.curator.framework.recipes.leader.LeaderSelector;
import org.apache.curator.framework.recipes.leader.LeaderSelectorListenerAdapter;
import org.apache.zookeeper.CreateMode;
import java.net.InetAddress;
import java.util.List;

public class ServiceRegistry {
    private static final String ELECTION_PATH = "/election";
    private static final String WORKERS_REGISTRY_PATH = "/workers";
    private final CuratorFramework client;

    public ServiceRegistry(CuratorFramework client) {
        this.client = client;
    }

    // Leader Election Logic
    public void startLeaderElection(Runnable onLeadershipGained) {
        LeaderSelector leaderSelector = new LeaderSelector(client, ELECTION_PATH, new LeaderSelectorListenerAdapter() {
            @Override
            public void takeLeadership(CuratorFramework client) throws Exception {
                onLeadershipGained.run();
                while (!Thread.currentThread().isInterrupted()) {
                    Thread.sleep(Long.MAX_VALUE);
                }
            }
        });
        leaderSelector.autoRequeue();
        leaderSelector.start();
    }

    // Service Registration Logic
    public void registerWorker(int port) throws Exception {
        if (client.checkExists().forPath(WORKERS_REGISTRY_PATH) == null) {
            client.create().creatingParentsIfNeeded().forPath(WORKERS_REGISTRY_PATH);
        }
        String ipAddress = InetAddress.getLocalHost().getHostAddress();
        String workerPath = WORKERS_REGISTRY_PATH + "/worker_" + ipAddress + ":" + port;
        client.create().withMode(CreateMode.EPHEMERAL).forPath(workerPath);
        System.out.println("[Registry] Worker registered at: " + ipAddress + ":" + port);
    }

    // Service Discovery Logic
    public List<String> getActiveWorkers() throws Exception {
        return client.getChildren().forPath(WORKERS_REGISTRY_PATH);
    }
}