package com.distributed.search;

import com.distributed.search.model.SearchTask;
import com.distributed.search.model.TaskResult;
import java.io.IOException;
import java.net.ServerSocket;
import java.net.Socket;
import java.util.Map;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.Executors;

/**
 * خادم العامل المعدل: يسمح للقائد بمعرفة المنفذ الخاص به لاستثنائه من العمل.
 */
public class WorkerServer {
    private final SearchEngine searchEngine;
    private final int port;
    private final ExecutorService threadPool;

    //  (Static) for current port
    private static int localBoundPort;

    public WorkerServer(SearchEngine searchEngine) throws IOException {
        this.searchEngine = searchEngine;

        // oper random port
        ServerSocket serverSocket = new ServerSocket(0);
        this.port = serverSocket.getLocalPort();

        // save port in static variable localboundport
        localBoundPort = this.port;

        this.threadPool = Executors.newCachedThreadPool();
        startListening(serverSocket);
    }

    /**
     * دالة ساكنة يستدعيها القائد (Coordinator) لمعرفة المنفذ الخاص به.
     */
    public static int getLocalBoundPort() {
        return localBoundPort;
    }

    private void startListening(ServerSocket serverSocket) {
        Thread serverThread = new Thread(() -> {
            System.out.println("[WorkerServer] Started and listening on port: " + port);
            try {
                while (!Thread.currentThread().isInterrupted()) {
                    Socket clientSocket = serverSocket.accept();
                    // معالجة كل طلب قادم في خيط (Thread) منفصل
                    threadPool.submit(() -> handleClientRequest(clientSocket));
                }
            } catch (IOException e) {
                if (!serverSocket.isClosed()) {
                    System.err.println("[WorkerServer] Server error: " + e.getMessage());
                }
            }
        });
        serverThread.setDaemon(true);
        serverThread.start();
    }

    private void handleClientRequest(Socket socket) {
        try (socket) {
            // استقبال المهمة (الكلمة والملفات) من القائد
            SearchTask task = SearchTask.parseDelimitedFrom(socket.getInputStream());

            if (task != null) {
                // ملاحظة: بما أن القائد لن يرسل لنفسه، هذه الرسالة ستظهر فقط عند العمال الآخرين
                System.out.println("[WorkerServer] Node " + port + " is processing " + task.getFilePathsCount() + " files...");

                // تنفيذ حساب التردد (TF)
                Map<String, Double> results = searchEngine.calculateTFForFiles(
                        task.getFilePathsList(),
                        task.getQuery()
                );

                // إرسال النتائج (TF scores) للقائد
                TaskResult response = TaskResult.newBuilder()
                        .putAllDocTfScores(results)
                        .build();
                response.writeDelimitedTo(socket.getOutputStream());
            }
        } catch (IOException e) {
            System.err.println("[WorkerServer] Error handling request: " + e.getMessage());
        }
    }

    public int getPort() {
        return port;
    }
}